import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle/index';
import WarningAlert from '../../projects/alerts/WarningAlert';
import AggrPointsTable from './AggrPointsTable';
import { wrapSubProjectPage } from '../../projects/wrapSubProjectPage';
import TimeRangeSelector from './TimeRangeSelector';
import Auth from '../../../auth/Auth';
import { ADMIN } from '../../../auth/Roles';
import StockpileKeysPanel from './StockpileKeysPanel';
import MoreDropdownButton from './MoreDropdownButton';
import LabelsSelector from './LabelsSelector';
import { computeRangeFromParams } from '../utils';
import { formatSearch } from '../../../utils/url';
import { formatISO8601 } from '../../../utils/iso8601';
import Selectors from '../../../utils/Selectors';

import { loadStockpileSensorData, clearStockpileSensorData } from '../../../store/reducers/data/stockpileSensorData';

class StockpileMetricPage extends PureComponent {
  static mapSearchToState(search) {
    const params = new URLSearchParams(search);

    const b = params.get('b') || '1d';
    const e = params.get('e') || '';

    const nowMillis = Date.now();

    const { fromMillis, toMillis } = computeRangeFromParams(b, e, nowMillis);

    const name = params.get('name') || '';

    const labels = params.get('labels') || '';

    const newFormatLabels = name ? `${Selectors._formatNameSelector(name)}{${labels}}` : labels;

    const forceCluster = params.get('forceCluster') || '';

    return {
      b, e, fromMillis, toMillis, labels: newFormatLabels, forceCluster, activePage: 1,
    };
  }

  static divideSelectorsToNameAndLabels(labels) {
    const search = {};

    try {
      const parsedLabels = Selectors.parse(labels);
      const name = parsedLabels.getNameSelector();
      if (name) {
        search.name = name;
      }
      search.labels = new Selectors('', parsedLabels.getSelectors()).format();
    } catch (e) {
      console.warn('failed to parse labels', e);
      search.labels = labels;
    }

    return search;
  }

  static mapStateToSearch(state) {
    const search = StockpileMetricPage.divideSelectorsToNameAndLabels(state.labels);

    if (state.b && state.b !== '1d') {
      search.b = state.b;
    }

    if (state.e) {
      search.e = state.e;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {
      ...StockpileMetricPage.divideSelectorsToNameAndLabels(state.labels),
      from: new Date(state.fromMillis).toISOString(),
      to: new Date(state.toMillis).toISOString(),
    };

    if (state.forceCluster) {
      params.forceCluster = state.forceCluster;
    }

    return params;
  }

  constructor(props) {
    super(props);

    this.state = StockpileMetricPage.mapSearchToState(this.props.location.search);
  }

  componentDidMount() {
    const { projectId } = this.props.match.params;
    const params = StockpileMetricPage.mapStateToParams(this.state);
    this.props.loadStockpileSensorData(projectId, params);
  }

  componentWillUnmount() {
    this.props.clearStockpileSensorData();
  }

  handleTimeRangeChange = ({ fromMillis, toMillis }) => {
    this.setState({
      b: formatISO8601(fromMillis),
      e: formatISO8601(toMillis),
      fromMillis,
      toMillis,
      activePage: 1,
    }, () => this.reloadData());
  };

  handleLabelsChange = (labels) => {
    this.setState({ labels, activePage: 1 }, () => this.reloadData());
  };

  reloadData = () => {
    const { projectId } = this.props.match.params;

    const search = StockpileMetricPage.mapStateToSearch(this.state);
    this.props.history.push(`?${new URLSearchParams(search).toString()}`);

    const params = StockpileMetricPage.mapStateToParams(this.state);
    this.props.loadStockpileSensorData(projectId, params);
  };

  handleActivePageChange = (activePage) => {
    this.setState({ activePage });
  };

  render() {
    const { match, stockpileSensorData } = this.props;
    const { projectId } = match.params;
    const { fromMillis, toMillis } = this.state;

    if (!Auth.hasRole(ADMIN)) {
      return (
        <WarningAlert
          title="Permisson denied!"
          message="You have not UI_GOD permissions to see this page"
        />
      );
    }

    let result;

    if (stockpileSensorData.loading) {
      result = <span>Loading...</span>;
    } else if (stockpileSensorData.error) {
      result = (
        <WarningAlert
          title="Stockpile metric error!"
          message={stockpileSensorData.error}
        />
      );
    } else {
      const stockpileSensor = stockpileSensorData.data;

      if (isEmpty(stockpileSensor)) {
        result = null;
      } else {
        let pointsTable;

        if (stockpileSensor.columns.length === 0 || stockpileSensor.points.length === 0) {
          pointsTable = <span>No points found</span>;
        } else {
          pointsTable = (
            <AggrPointsTable
              columns={stockpileSensor.columns}
              points={stockpileSensor.points}
              activePage={this.state.activePage}
              onChangeActivePage={this.handleActivePageChange}
            />
          );
        }

        const keyPanel = (
          <StockpileKeysPanel
            projectId={projectId}
            labels={this.state.labels}
            b={this.state.b}
            e={this.state.e}
            fromMillis={this.state.fromMillis}
            toMillis={this.state.toMillis}
            forceCluster={this.state.forceCluster}
            keys={stockpileSensor.keys}
          />
        );

        result = (
          <div>
            <p>
              Kind:
              {' '}
              {stockpileSensor.kind}
            </p>
            {keyPanel}
            {pointsTable}
          </div>
        );
      }
    }

    // eslint-disable-next-line max-len
    const urlParams = new URLSearchParams(StockpileMetricPage.mapStateToParams(this.state)).toString();

    return (
      <div>
        <Breadcrumb match={match} />
        <PageTitle title="Stockpile metric" />
        <div className="form">
          <div className="form-group">
            <LabelsSelector labels={this.state.labels} onChange={this.handleLabelsChange} />
          </div>
          <div className="form-group">
            <div className="form-inline">
              <div className="form-group">
                <TimeRangeSelector
                  fromMillis={fromMillis}
                  toMillis={toMillis}
                  onChange={this.handleTimeRangeChange}
                />
              </div>
              <div className="form-group pull-right">
                <MoreDropdownButton
                  projectId={projectId}
                  state={this.state}
                  urlParams={urlParams}
                  stockpileMetric={stockpileSensorData.data}
                />
              </div>
            </div>
          </div>
        </div>
        <div>
          {stockpileSensorData.keys}
        </div>
        <div>
          {result}
        </div>
      </div>
    );
  }
}

StockpileMetricPage.propTypes = {
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  stockpileSensorData: PropTypes.object.isRequired,
  loadStockpileSensorData: PropTypes.func.isRequired,
  clearStockpileSensorData: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  stockpileSensorData: state.stockpileSensorData,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadStockpileSensorData,
  clearStockpileSensorData,
}, dispatch);

const component = connect(mapStateToProps, mapDispatchToProps)(StockpileMetricPage);

export default wrapSubProjectPage()(component);
