import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { formatLocalDate, parseLocalDate } from '../../../utils/localDate';

class TimeRangeSelector extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      from: formatLocalDate(new Date(props.fromMillis)),
      to: formatLocalDate(new Date(props.toMillis)),
    };
  }

  componentDidUpdate(prevProps) {
    if (this.props.fromMillis !== prevProps.fromMillis
      || this.props.toMillis !== prevProps.toMillis) {
      this.setState({
        from: formatLocalDate(new Date(this.props.fromMillis)),
        to: formatLocalDate(new Date(this.props.toMillis)),
      });
    }
  }

  handleInputChange = (event) => {
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  handleKeyPress = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.handleChange();
    }
  };

  handleSubmitClick = () => {
    this.handleChange();
  };

  handleChange = () => {
    const fromDate = parseLocalDate(this.state.from);
    const toDate = parseLocalDate(this.state.to);

    if (fromDate && toDate) {
      const fromMillis = fromDate.getTime();
      const toMillis = toDate.getTime();
      this.props.onChange({ fromMillis, toMillis });
    }
  };

  render() {
    return (
      <div className="form-inline">
        <div className="form-group">
          <input
            className="form-control"
            type="text"
            name="from"
            value={this.state.from}
            onChange={this.handleInputChange}
            onKeyPress={this.handleKeyPress}
          />
        </div>
        <div className="form-group" style={{ padding: '0 5px' }}>
          &mdash;
        </div>
        <div className="form-group">
          <input
            className="form-control"
            type="text"
            name="to"
            value={this.state.to}
            onChange={this.handleInputChange}
            onKeyPress={this.handleKeyPress}
          />
        </div>
        <button
          type="button"
          className="btn btn-primary"
          style={{ marginLeft: '5px' }}
          onClick={this.handleSubmitClick}
        >
          Apply
        </button>
      </div>
    );
  }
}

TimeRangeSelector.propTypes = {
  fromMillis: PropTypes.number.isRequired,
  toMillis: PropTypes.number.isRequired,
  onChange: PropTypes.func.isRequired,
};

export default TimeRangeSelector;
