import isEmpty from 'lodash/isEmpty';
import { formatLocalDate } from '../../../utils/localDate';

export function downloadFile(filename, content) {
  const $a = document.createElement('a');
  const url = `data:text/plain;charset=utf-8,${encodeURIComponent(content)}`;
  $a.setAttribute('href', url);
  $a.setAttribute('download', filename);

  if (document.createEvent) {
    const event = document.createEvent('MouseEvents');
    event.initEvent('click', true, true);
    $a.dispatchEvent(event);
  } else if ($a.click) {
    $a.click();
  }
}

function getAggrPointValue(aggrPoint, column) {
  switch (column) {
    case 'TS': {
      const value = aggrPoint.tsMillis || 0;
      return formatLocalDate(new Date(value));
    }
    case 'VALUE_NUM': {
      return aggrPoint.valueNum || 0;
    }
    case 'VALUE_DENOM': {
      return aggrPoint.valueDenom || 0;
    }
    case 'MERGE': {
      return aggrPoint.merge || false;
    }
    case 'COUNT': {
      return aggrPoint.count || 0;
    }
    case 'STEP': {
      return aggrPoint.stepMillis || 0;
    }
    case 'HISTOGRAM': {
      const value = aggrPoint.histogram || {};
      return JSON.stringify(value);
    }
    case 'LOG_HISTOGRAM': {
      const value = aggrPoint.logHistogram || {};
      return JSON.stringify(value);
    }
    case 'ISUMMARY': {
      const value = aggrPoint.summaryInt64 || {};
      return JSON.stringify(value);
    }
    case 'DSUMMARY': {
      const value = aggrPoint.summaryDouble || {};
      return JSON.stringify(value);
    }
    case 'LONG_VALUE': {
      return aggrPoint.longValue || 0;
    }
    default: {
      return '-';
    }
  }
}

export function parseStockpileMetricAsTable(points, columns) {
  const newColumns = [...columns];

  const valueColIndex = columns.indexOf('VALUE');

  if (valueColIndex >= 0) {
    if (points.some((p) => p.valueDenom)) {
      newColumns.splice(valueColIndex, 1, 'VALUE_NUM', 'VALUE_DENOM');
    } else {
      newColumns.splice(valueColIndex, 1, 'VALUE_NUM');
    }
  }

  const rows = points.map((p) => newColumns.map((c) => getAggrPointValue(p, c)));
  return { columns: newColumns, rows };
}

const FORBIDDEN_CSV_SYMBOLS = /(,|\\s|")/;

function quoteCsvValue(value) {
  if (FORBIDDEN_CSV_SYMBOLS.test(value)) {
    return `"${value.replace('"', '""')}"`;
  }
  return value;
}

export function convertToCsv(stockpileMetric) {
  if (isEmpty(stockpileMetric)) {
    return '';
  }

  const { points, columns } = stockpileMetric;

  if (columns.length === 0 || points.length === 0) {
    return '';
  }

  const table = parseStockpileMetricAsTable(points, columns);

  let csv = `${table.columns.join(',')}\r\n`;

  csv += table.rows.map((row) => row.map((value) => quoteCsvValue(value)).join(',')).join('\r\n');
  csv += '\r\n';

  return csv;
}
