/* eslint-disable react/no-array-index-key,max-len */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';

import compareTextWithNumber from '../../../utils/sortTextWithNumber';

import './style.css';

const ROW_SHAPE = PropTypes.shape({
  color: PropTypes.string.isRequired,
  names: PropTypes.arrayOf(PropTypes.string).isRequired,
  values: PropTypes.arrayOf(PropTypes.number).isRequired,
  formattedValues: PropTypes.arrayOf(PropTypes.string).isRequired,
});

const SORT_DIRECTIONS = {
  UNSORTED: 0,
  ASC: -1,
  DESC: 1,
};

const COLUMN_TYPES = {
  NAME: 0,
  VALUE: 1,
};

class TableLegend extends PureComponent {
  static _mapSortDirectionToClassName(sortDirection) {
    switch (sortDirection) {
      case SORT_DIRECTIONS.UNSORTED:
        return 'unsorted';
      case SORT_DIRECTIONS.ASC:
        return 'asc';
      case SORT_DIRECTIONS.DESC:
        return 'desc';
      default:
        throw new Error(`unknown sort direction: ${sortDirection}`);
    }
  }

  static _sortRows(rows, columnType, columnId, sortDirection) {
    if (sortDirection === 0) {
      return rows;
    }

    if (columnType === COLUMN_TYPES.NAME) {
      return rows.sort((a, b) => {
        const aName = a.names[columnId];
        const bName = b.names[columnId];

        return sortDirection * compareTextWithNumber(aName, bName);
      });
    }

    if (columnType === COLUMN_TYPES.VALUE) {
      return rows.sort((a, b) => {
        const aValue = a.values[columnId];
        const bValue = b.values[columnId];

        if (isNaN(aValue) && isNaN(bValue)) {
          return 0;
        } if (isNaN(aValue)) {
          return 1;
        } if (isNaN(bValue)) {
          return -1;
        }

        return sortDirection * (aValue - bValue);
      });
    }

    throw new Error(`unknown column type: ${columnType}`);
  }

  constructor(props) {
    super(props);

    this.state = {
      rows: props.rows,
      sortedRows: props.rows,
      columnType: COLUMN_TYPES.NAME,
      columnId: 0,
      sortDirection: SORT_DIRECTIONS.UNSORTED,
    };
  }

  componentDidUpdate(prevProps) {
    const { rows } = this.props;
    const prevRows = prevProps.rows;

    if (rows !== prevRows) {
      const sortedRows = TableLegend._sortRows(
        rows,
        this.state.columnType,
        this.state.columnId,
        this.state.sortDirection,
      );

      this.setState({ rows, sortedRows });
    }
  }

  handleHeaderClick = (event) => {
    event.preventDefault();

    const dataSet = event.currentTarget.dataset;

    if (isEmpty(dataSet)) {
      return;
    }

    let { columnType, columnId } = dataSet;

    columnType = parseInt(columnType, 10);
    columnId = parseInt(columnId, 10);

    let sortDirection;

    if (this.state.columnType === columnType && this.state.columnId === columnId) {
      sortDirection = this.state.sortDirection === 0
        ? SORT_DIRECTIONS.DESC
        : -this.state.sortDirection;
    } else {
      sortDirection = SORT_DIRECTIONS.DESC;
    }

    const sortedRows = TableLegend._sortRows(this.state.rows, columnType, columnId, sortDirection);

    this.setState({
      sortedRows, columnType, columnId, sortDirection,
    });
  };

  _getSortDirectionClassName = (columnType, columnId) => {
    let sortDirection;

    if (this.state.columnType === columnType && this.state.columnId === columnId) {
      sortDirection = this.state.sortDirection;
    } else {
      sortDirection = SORT_DIRECTIONS.UNSORTED;
    }

    return TableLegend._mapSortDirectionToClassName(sortDirection);
  };

  render() {
    const { nameTitles, valueTitles } = this.props;
    const { sortedRows } = this.state;

    if (sortedRows.length === 0) {
      return null;
    }

    return (
      <table className="table table-condensed table-bordered table-striped table-legend">
        <thead>
          <tr>
            <th className="legend-color-name" />
            {nameTitles.map((nameTitle, nameIndex) => {
              const sortDirectionClassName = this._getSortDirectionClassName(COLUMN_TYPES.NAME, nameIndex);

              return (
                <th
                  key={`name-${nameTitle}`}
                  className={sortDirectionClassName}
                  data-column-type={COLUMN_TYPES.NAME}
                  data-column-id={nameIndex}
                  onClick={this.handleHeaderClick}
                >
                  <div className="header-inner">{nameTitle}</div>
                </th>
              );
            })}
            {valueTitles.map((valueTitle, valueIndex) => {
              const sortDirectionClassName = this._getSortDirectionClassName(COLUMN_TYPES.VALUE, valueIndex);

              return (
                <th
                  key={`value-${valueTitle}`}
                  className={`table-legend__value-column ${sortDirectionClassName}`}
                  data-column-type={COLUMN_TYPES.VALUE}
                  data-column-id={valueIndex}
                  onClick={this.handleHeaderClick}
                >
                  <div className="header-inner">{valueTitle}</div>
                </th>
              );
            })}
          </tr>
        </thead>
        <tbody>
          {sortedRows.map((row) => (
            <tr className="table-legend__row" key={`row-${row.names.join(';')}`}>
              <td>
                <div className="solomon-color-square">
                  <div style={{ backgroundColor: row.color }} />
                </div>
              </td>
              {row.names.map((name) => (
                <td style={{ whiteSpace: 'nowrap' }} key={`name-${name}`}>
                  {name}
                </td>
              ))}
              {row.formattedValues.map((value, valueIndex) => (
                <td className="table-legend__value" key={`value-${valueTitles[valueIndex]}`}>
                  {row.formattedValues[valueIndex]}
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    );
  }
}

TableLegend.propTypes = {
  nameTitles: PropTypes.array.isRequired,
  valueTitles: PropTypes.array.isRequired,
  rows: PropTypes.arrayOf(ROW_SHAPE).isRequired,
};

export default TableLegend;
