import { parseISO8601 } from '../../utils/iso8601';
import { parseDuration } from '../../utils/duration';
import parseBoolean from '../../utils/boolean';
import { DEFAULT_NUMBER_FORMAT_VALUE } from '../graphs/NumberFormatUtils';
import TimeFormatUtils from '../../utils/time/TimeFormatUtils';

function computeInstantFromParam(bName, b, nowMillis, supportOldFormat) {
  if (!b || b === '0') {
    return nowMillis;
  }

  const duration = parseDuration(b);

  if (duration) {
    return nowMillis - duration;
  }

  const time = parseISO8601(b);

  if (!isNaN(time)) {
    return time;
  }

  if (supportOldFormat) {
    const oldDate = TimeFormatUtils.parseShortDateTimeInLocalTz(b);
    if (oldDate !== null) {
      return oldDate.getTime();
    }
  }

  throw new Error(`failed to parse time range parameter: ${bName}=${b}`);
}

export function computeRangeFromParams(b, e, nowMillis, supportOldFormat = false) {
  let fromMillis = computeInstantFromParam('b', b, nowMillis, supportOldFormat);
  let toMillis = computeInstantFromParam('e', e, nowMillis, supportOldFormat);

  if (fromMillis > toMillis) {
    const temp = fromMillis;
    fromMillis = toMillis;
    toMillis = temp;
  }

  return { fromMillis, toMillis };
}

export function computeRangeFromParamsForOldUi(b, e, nowMillis) {
  return computeRangeFromParams(b, e, nowMillis, true);
}

/**
 * @param {URLSearchParams} search
 */
export function mapSearchToAutoGraphConfig(search) {
  const config = {};

  config.stack = parseBoolean(search.get('stack') || '');
  config.norm = parseBoolean(search.get('norm') || '');
  config.interpolate = search.get('interpolate') || '';

  config.dropNaN = parseBoolean(search.get('dropNaN' || ''));

  const downsamplingMode = search.get('downsamplingMode') || 'auto';
  config.downsamplingMode = downsamplingMode;

  if (downsamplingMode === 'byPoints') {
    config.maxPoints = parseInt(search.get('maxPoints') || '', 10);
  } else if (downsamplingMode === 'byGrid') {
    config.grid = search.get('grid') || '1h';
  }

  config.downsamplingAggr = search.get('downsamplingAggr') || 'default';
  config.downsamplingFill = search.get('downsamplingFill') || 'null';

  config.max = search.get('max') || '';
  config.min = search.get('min') || '';
  config.numberFormat = search.get('numberFormat') || '';
  config.scale = search.get('scale') || 'natural';

  config.cs = search.get('cs') || 'auto';
  config.green = search.get('green') || '';
  config.yellow = search.get('yellow') || '';
  config.red = search.get('red') || '';
  config.violet = search.get('violet') || '';

  return config;
}

export function mapAutoGraphConfigToSearch(config) {
  const search = {};

  if (config.stack) {
    search.stack = config.stack;
  }

  if (config.norm) {
    search.norm = config.norm;
  }

  if (config.interpolate && config.interpolate !== 'linear') {
    search.interpolate = config.interpolate;
  }

  if (config.dropNaN) {
    search.dropNaN = config.dropNaN;
  }

  if (config.downsamplingMode !== 'auto') {
    search.downsamplingMode = config.downsamplingMode;
  }

  if (config.downsamplingMode === 'byPoints') {
    search.maxPoints = config.maxPoints || 1000;
  }

  if (config.downsamplingMode === 'byGrid') {
    search.grid = config.grid || '1h';
  }

  if (config.downsamplingAggr !== 'default') {
    search.downsamplingAggr = config.downsamplingAggr;
  }

  if (config.downsamplingFill !== 'null') {
    search.downsamplingFill = config.downsamplingFill;
  }

  if (config.max) {
    search.max = config.max;
  }

  if (config.min) {
    search.min = config.min;
  }

  if (config.numberFormat && config.numberFormat !== DEFAULT_NUMBER_FORMAT_VALUE) {
    search.numberFormat = config.numberFormat;
  }

  if (config.scale !== 'natural') {
    search.scale = config.scale;
  }

  if (config.cs && config.cs !== 'auto') {
    search.cs = config.cs;

    if (config.cs === 'gradient') {
      if (config.green) {
        search.green = config.green;
      }

      if (config.yellow) {
        search.yellow = config.yellow;
      }

      if (config.red) {
        search.red = config.red;
      }

      if (config.violet) {
        search.violet = config.violet;
      }
    }
  }

  return search;
}

export function mapAutoGraphConfigToDownsamplingParams(config) {
  const downsampling = {};

  if (config.downsamplingMode === 'off') {
    downsampling.disabled = true;
    return downsampling;
  }

  if (config.downsamplingMode === 'auto') {
    downsampling.maxPoints = 1000;
  }

  if (config.downsamplingMode === 'byPoints') {
    downsampling.maxPoints = config.maxPoints || 1000;
  }

  if (config.downsamplingMode === 'byGrid') {
    downsampling.gridMillis = parseDuration(config.grid || '1h');
  }

  if (config.downsamplingAggr !== 'default') {
    downsampling.aggregation = config.downsamplingAggr.toUpperCase();
  }

  if (config.downsamplingFill !== 'null') {
    downsampling.fill = config.downsamplingFill.toUpperCase();
  }

  return downsampling;
}
