import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import TableToolbar from '../../components/TableToolbar';
import Pagination from '../../components/Pagination';

import { formatSearch, parseSearch } from '../../utils/url';

import { findAllGraphs } from '../../store/reducers/graphs/globalGraphsPage';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const { text, page, pageSize } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  return params;
};

class GlobalGraphsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findAllGraphs(cleanState(this.state));
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findAllGraphs(newCleanedState);
  }

  render() {
    const { graphsPage } = this.props;
    const graphs = graphsPage.result || [];
    const pagination = graphsPage.page;
    const indexOffset = (pagination !== undefined)
      ? pagination.pageSize * pagination.current
      : 0;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title="Graphs" />

        <TableToolbar
          filter={this.state.text}
          onFilterChange={this.onFilterChange}
        />

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Id</th>
              <th>Project</th>
              <th>Name</th>
            </tr>
          </thead>
          <tbody>
            {graphs.map((c, i) => (
              <tr key={c.id}>
                <td>{i + indexOffset + 1}</td>
                <td>
                  <Link to={`/admin/projects/${c.projectId}/graphs/${c.id}`}>{c.id}</Link>
                </td>
                <td>
                  <Link to={`/admin/projects/${c.projectId}`}>{c.projectId}</Link>
                </td>
                <td>{c.name}</td>
              </tr>
            ))}
          </tbody>
        </table>
        {pagination && (
          <Pagination
            activePage={pagination.current + 1}
            pageCount={pagination.pagesCount}
            pageSize={pagination.pageSize}
            totalCount={pagination.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        )}
      </div>
    );
  }
}

GlobalGraphsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  graphsPage: PropTypes.object.isRequired,
  findAllGraphs: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  graphsPage: state.globalGraphsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findAllGraphs,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(GlobalGraphsPage);
