import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import GraphForm from './GraphForm';

import { parseSearch } from '../../utils/url';
import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { findGraph, saveGraph, clearGraph } from '../../store/reducers/graphs/graph';

class GraphEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._graphId = props.match.params.graphId;
    this._isNew = !this._graphId;
    const parsedSearch = parseSearch(props.location.search);

    if (this._isNew) {
      this._graphId = parsedSearch.get('duplicateOf');
      this._isFromAutoGraph = props.graph ? !!props.graph._isFromAutoGraph : false;
    }
  }

  componentDidMount() {
    const { graph } = this.props;
    if (this._graphId) {
      const loadedId = graph.id;
      if (loadedId !== this._graphId) {
        this.props.findGraph(this._projectId, this._graphId);
      }
    } else if (!this._isFromAutoGraph) {
      this.props.clearGraph();
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveGraph(this._projectId, value, this._isNew)
      .then(() => history.push(`/admin/projects/${this._projectId}/graphs/${value.id}`));
  };

  render() {
    const { graph } = this.props;
    const title = this._isNew ? 'New Graph' : `Graph ${this._graphId}`;
    const isLoading = this._graphId && isEmpty(graph);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
        && (
        <GraphForm
          projectId={this._projectId}
          graph={graph}
          isNew={this._isNew}
          isFromAutoGraph={this._isFromAutoGraph}
          onSubmit={this.onSubmit}
        />
        )}
      </div>
    );
  }
}

GraphEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  graph: PropTypes.object.isRequired,
  findGraph: PropTypes.func.isRequired,
  saveGraph: PropTypes.func.isRequired,
  clearGraph: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({ graph: state.graph });

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findGraph,
  saveGraph,
  clearGraph,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(GraphEditPage);

export default wrapSubProjectEditPage(connectedPage);
