import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import FormElement from '../../components/forms/FormElement';
import ApiCallExample from '../../components/ApiCallExample';
import { EditableTable } from '../../components/ParameterTables';
import { inputTargetValue } from '../../utils/inputs';
import GraphElementEditor from '../../components/GraphElementEditor';
import FormCheckbox from '../../components/forms/FormCheckbox';
import FormArea from '../../components/forms/FormArea';
import NumberFormatSelector from './NumberFormatSelector';
import { TabPane, Tabs } from '../../components/Tabs';
import FormGroup from '../../components/forms/FormGroup';
import FormSelect from '../../components/forms/FormSelect';
import * as SETTINGS from './settings';
import FormRadioGroup from './FormRadioGroup';

class GraphForm extends PureComponent {
  static graphToState = (graph, isNew, isFromAutoGraph) => {
    if (isFromAutoGraph || graph.id) {
      const state = { ...graph };
      delete state._isFromAutoGraph;
      state.id = isNew ? '' : state.id;
      state.name = isNew ? '' : state.name;
      return state;
    }
    return { id: '', name: '' };
  };

  constructor(props) {
    super(props);
    this.state = GraphForm.graphToState(props.graph, props.isNew, props.isFromAutoGraph);
    this._nameInputTouched = !!props.graph.name;
    this._isNewGraph = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId || this.props.graph !== prevProps.graph) {
      this.state = GraphForm.graphToState(this.props.graph, this.props.isFromAutoGraph);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);

    const change = {};
    switch (target.name) {
      case 'id':
        change.id = value;
        if (!this._nameInputTouched) {
          change.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        change.name = value;
        break;
      default:
        change[target.name] = value;
    }

    this.setState(change);
  };

  onNumberInputChange = (event) => {
    this.setState({ [event.target.name]: event.target.value });
  };

  onSelectChange = (event) => {
    // Don't use event.preventDefault(), because event is fake object
    this.setState({ [event.target.name]: event.target.value });
  };

  onBoolSelectChange = (event) => {
    const { name } = event.target;

    let value;
    switch (event.target.value) {
      case 'true':
        value = true;
        break;
      case 'false':
        value = false;
        break;
      default:
        value = null;
    }

    this.setState({ [name]: value });
  };

  onNumberFormatChange = (numberFormat) => {
    this.setState({ numberFormat });
  };

  onParameterUpdate = (paramIndex, param) => {
    const parameters = this.state.parameters ? [...this.state.parameters] : [];
    if (paramIndex < 0) {
      parameters.push(param);
    } else {
      parameters.splice(paramIndex, 1, param);
    }
    this.setState({ parameters });
  };

  onParameterDelete = (paramIndex) => {
    const parameters = this.state.parameters ? [...this.state.parameters] : [];
    parameters.splice(paramIndex, 1);
    this.setState({ parameters });
  };

  onElementsChange = (elements) => {
    this.setState({ elements });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.state);
  };

  onJsonStateChange = (newState) => {
    this.setState(newState);
  };

  _generalTabPane() {
    return (
      <TabPane label="General">
        <FormInput
          type="text"
          name="id" label="Id"
          value={this.state.id}
          onChange={this.onInputChange}
          help="Unique graph identificator"
          disabled={!this._isNewGraph}
        />
        <FormInput
          type="text"
          name="name" label="Name"
          value={this.state.name}
          onChange={this.onInputChange}
          help="This value will be displayed in Solomon UI"
        />
        <FormArea
          name="description"
          label="Description"
          value={this.state.description}
          onChange={this.onInputChange}
          help="Graph short description. It will be displayed near graph title"
        />
      </TabPane>
    );
  }

  _sensorsTabPane() {
    return (
      <TabPane label="Metrics">
        <FormElement label="Parameters">
          <EditableTable
            parameters={this.state.parameters || []}
            onDelete={this.onParameterDelete}
            onUpdate={this.onParameterUpdate}
          />
        </FormElement>
        <FormElement label="Elements">
          <GraphElementEditor
            elements={this.state.elements}
            onChange={this.onElementsChange}
          />
        </FormElement>
      </TabPane>
    );
  }

  _displayTabPane() {
    const graphMode = this.state.graphMode || 'GRAPH';
    const secondaryGraphMode = this.state.secondaryGraphMode || 'PIE';
    const hasYaxis = SETTINGS.checkModeHasYaxis(graphMode)
      || SETTINGS.checkModeHasYaxis(secondaryGraphMode);
    const hasHeatmap = graphMode === 'HEATMAP'
      || secondaryGraphMode === 'HEATMAP';

    return (
      <TabPane label="Display">
        <FormGroup title="Modes">
          <FormSelect
            name="graphMode"
            label="Main graph mode"
            value={graphMode}
            options={SETTINGS.GRAPH_MODE_OPTIONS}
            onChange={this.onSelectChange}
          />
          <FormSelect
            name="secondaryGraphMode"
            label="Secondary graph mode"
            value={secondaryGraphMode}
            options={SETTINGS.SECONDARY_GRAPH_MODE_OPTIONS}
            onChange={this.onSelectChange}
          />
        </FormGroup>
        {hasHeatmap && (
          <FormGroup title="Heatmap thresholds">
            <FormInput
              type="text"
              name="greenValue" label="Green"
              value={this.state.greenValue}
              onChange={this.onInputChange}
            />
            <FormInput
              type="text"
              name="yellowValue" label="Yellow"
              value={this.state.yellowValue}
              onChange={this.onInputChange}
            />
            <FormInput
              type="text"
              name="redValue" label="Red"
              value={this.state.redValue}
              onChange={this.onInputChange}
            />
            <FormInput
              type="text"
              name="violetValue" label="Violet"
              value={this.state.violetValue}
              onChange={this.onInputChange}
            />
          </FormGroup>
        )}
        {graphMode === 'GRAPH' && (
          <FormGroup title="Stacking">
            <FormRadioGroup
              name="stack" label="Stack"
              value={String(this.state.stack)}
              defaultValue="null"
              options={SETTINGS.STACK_OPTIONS}
              onChange={this.onBoolSelectChange}
            />
            <FormCheckbox
              name="normalize"
              label="Normalize"
              value={this.state.normalize}
              onChange={this.onInputChange}
            />
          </FormGroup>
        )}
        <FormGroup title="Color scheme">
          <FormRadioGroup
            name="colorScheme" label="Color scheme"
            value={this.state.colorScheme} defaultValue="AUTO"
            options={SETTINGS.COLOR_SCHEME_OPTIONS}
            onChange={this.onSelectChange}
          />
          {this.state.colorScheme === 'GRADIENT' && (
            <div>
              <FormInput
                type="text"
                name="green" label="Green"
                value={this.state.green}
                onChange={this.onInputChange}
              />
              <FormInput
                type="text"
                name="yellow" label="Yellow"
                value={this.state.yellow}
                onChange={this.onInputChange}
              />
              <FormInput
                type="text"
                name="red" label="Red"
                value={this.state.red}
                onChange={this.onInputChange}
              />
              <FormInput
                type="text"
                name="violet" label="Violet"
                value={this.state.violet}
                onChange={this.onInputChange}
              />
            </div>
          )}
        </FormGroup>
        <FormGroup title="Y-axis">
          {hasYaxis && (
            <FormRadioGroup
              name="scale" label="Scale"
              value={this.state.scale} defaultValue="NATURAL"
              options={SETTINGS.SCALE_OPTIONS}
              onChange={this.onSelectChange}
            />
          )}
          <FormElement label="Number format">
            <NumberFormatSelector
              value={this.state.numberFormat}
              onChange={this.onNumberFormatChange}
            />
          </FormElement>
          {hasYaxis && (
            <div className="form-group">
              <label className="col-sm-2 control-label" htmlFor="minInput">Min</label>
              <div className="col-sm-4">
                <input
                  id="minInput"
                  className="form-control"
                  type="text"
                  name="min"
                  value={this.state.min}
                  onChange={this.onInputChange}
                />
              </div>
              <label className="col-sm-2 control-label" htmlFor="maxInput">Max</label>
              <div className="col-sm-4">
                <input
                  id="maxInput"
                  className="form-control"
                  type="text"
                  name="max"
                  value={this.state.max}
                  onChange={this.onInputChange}
                />
              </div>
            </div>
          )}
        </FormGroup>
        <FormGroup title="Others">
          <FormRadioGroup
            name="aggr" label="Aggregation type"
            value={this.state.aggr} defaultValue="AVG"
            options={SETTINGS.AGGR_OPTIONS}
            onChange={this.onSelectChange}
          />
          <FormRadioGroup
            name="interpolate" label="Interpolate"
            value={this.state.interpolate} defaultValue="LINEAR"
            options={SETTINGS.INTERPOLATE_OPTIONS}
            onChange={this.onSelectChange}
          />
          <FormRadioGroup
            name="dropNans"
            label="Drop NaNs"
            value={String(this.state.dropNans)}
            defaultValue="null"
            options={SETTINGS.DROP_NANS_OPTIONS}
            onChange={this.onBoolSelectChange}
          />
          <FormCheckbox
            name="hideNoData"
            label="Hide no data"
            value={this.state.hideNoData}
            onChange={this.onInputChange}
          />
          <FormInput
            type="text"
            name="threshold"
            label="Threshold"
            value={this.state.threshold}
            onChange={this.onNumberInputChange}
          />
        </FormGroup>
      </TabPane>
    );
  }

  _transformationsTabPane() {
    const transform = this.state.transform || 'NONE';
    const overLinesTransform = this.state.overLinesTransform || 'NONE';
    const filter = this.state.filter || 'NONE';
    const filterBy = this.state.filterBy || 'AVG';
    const downsampling = this.state.downsampling || 'AUTO';

    return (
      <TabPane label="Transformations">
        <FormGroup title="Transform">
          <FormSelect
            name="transform"
            label="Transform"
            value={transform}
            options={SETTINGS.TRANSFORM_OPTIONS}
            onChange={this.onSelectChange}
          />
          {['MOVING_AVERAGE', 'MOVING_PERCENTILE'].indexOf(overLinesTransform) >= 0 && (
            <FormInput
              type="text"
              name="movingWindow" label="Moving window"
              placeholder="5m"
              value={this.state.movingWindow}
              onChange={this.onInputChange}
            />
          )}
          {transform === 'MOVING_PERCENTILE' && (
            <FormInput
              type="text"
              name="movingWindow" label="Moving percentile"
              placeholder="80"
              value={this.state.movingPercentile}
              onChange={this.onInputChange}
            />
          )}
        </FormGroup>
        <FormGroup title="Overline transform">
          <FormSelect
            name="overLinesTransform"
            label="Overline transform"
            value={overLinesTransform}
            options={SETTINGS.OVERLINE_TRANSFORM_OPTIONS}
            onChange={this.onSelectChange}
          />
          {['PERCENTILE', 'WEIGHTED_PERCENTILE'].indexOf(overLinesTransform) >= 0 && (
            <FormInput
              type="text"
              name="percentiles" label="Percentiles"
              placeholder="50,90,99"
              value={this.state.percentiles}
              onChange={this.onInputChange}
            />
          )}
          {(overLinesTransform === 'WEIGHTED_PERCENTILE') && (
            <FormInput
              type="text"
              name="bucketLabel"
              label="Bucket label"
              placeholder="bin"
              value={this.state.bucketLabel}
              onChange={this.onInputChange}
            />
          )}
        </FormGroup>
        <FormGroup title="Filter">
          <FormSelect
            name="filter"
            label="Filter"
            value={filter}
            options={SETTINGS.FILTER_OPTIONS}
            onChange={this.onSelectChange}
          />
          {filter !== 'NONE' && (
            <FormSelect
              name="filterBy"
              label="Filter by"
              value={filterBy}
              options={SETTINGS.AGGR_OPTIONS}
              onChange={this.onSelectChange}
            />
          )}
          {filter !== 'NONE' && (
            <FormInput
              type="number" name="filterLimit" label="Filter limit"
              value={this.state.filterLimit}
              onChange={this.onInputChange}
            />
          )}
        </FormGroup>
        <FormGroup title="Downsampling">
          <FormSelect
            name="downsampling"
            label="Mode"
            value={downsampling}
            defaultValue="AUTO"
            options={SETTINGS.DOWNSAMPLING_OPTIONS}
            onChange={this.onSelectChange}
          />
          {downsampling !== 'OFF' && (
            <FormSelect
              name="downsamplingAggr"
              label="Aggregation function"
              value={this.state.downsamplingAggr}
              defaultValue="DEFAULT"
              options={SETTINGS.DOWNSAMPLING_AGGR_OPTIONS}
              onChange={this.onSelectChange}
            />
          )}
          {downsampling !== 'OFF' && (
            <FormSelect
              name="downsamplingFill"
              label="Fill option"
              value={this.state.downsamplingFill}
              defaultValue="NULL"
              options={SETTINGS.DOWNSAMPLING_FILL_OPTIONS}
              onChange={this.onSelectChange}
            />
          )}
          {downsampling === 'BY_INTERVAL' && (
            <FormInput
              type="text"
              name="grid" label="Grid interval"
              placeholder="1m30s"
              value={this.state.grid}
              onChange={this.onInputChange}
            />
          )}
          {downsampling === 'BY_POINTS' && (
            <FormInput
              type="number"
              name="maxPoints" label="Max points"
              placeholder="100"
              value={this.state.maxPoints}
              onChange={this.onInputChange}
            />
          )}
        </FormGroup>
      </TabPane>
    );
  }

  render() {
    const { projectId } = this.props;

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <Tabs selected={1}>
            {this._generalTabPane()}
            {this._sensorsTabPane()}
            {this._displayTabPane()}
            {this._transformationsTabPane()}
          </Tabs>
          <FormButtons onSubmit={this.onSubmit} />
        </div>

        <div className="col-lg-6 col-md-5">
          <ApiCallExample
            path={`/projects/${projectId}/graphs`}
            value={this.state}
            isNew={this._isNewGraph}
            onChange={this.onJsonStateChange}
          />
        </div>
      </form>
    );
  }
}

GraphForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  graph: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  isFromAutoGraph: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default GraphForm;
