import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabeledValue from '../../components/LabeledValue';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';
import { ReadonlyTable } from '../../components/ParameterTables';
import GraphElementEditor from '../../components/GraphElementEditor';
import MonitoringButton from '../old/components/ControlBar/MonitoringButton';
import PreviewLink, { makeOldUiParamSelectors } from '../../components/PreviewLink/PreviewLink';

import * as SETTINGS from './settings';
import FormGroup from '../../components/forms/FormGroup';
import NumberFormatReadonly from './NumberFormatReadonly';
import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

import { deleteGraph, findGraph } from '../../store/reducers/graphs/graph';
import { fetchGraphParamSensors } from '../../store/reducers/paramSensors/graphParamSensorsPage';
import { isPreOrProdIntEnv } from '../../utils/env';

const getTitleByValue = (value, options) => {
  const filteredOptions = options.filter((o) => o.value === value || o.value === String(value));
  if (filteredOptions.length !== 1) {
    return '-';
  }
  return filteredOptions[0].title;
};

class GraphPage extends PureComponent {
  constructor(props) {
    super(props);
    const { projectId, graphId } = props.match.params;
    this._projectId = projectId;
    this._graphId = graphId;
  }

  componentDidMount() {
    this.props.findGraph(this._projectId, this._graphId)
      .then((resp) => {
        if (resp && resp.payload) {
          this.props.fetchGraphParamSensors(this._projectId, resp.payload.parameters);
        }
      });
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;
    if (params.graphId !== prevParams.graphId) {
      this._projectId = params.projectId;
      this._graphId = params.graphId;
      this.props.findGraph(this._projectId, this._projectId)
        .then((resp) => {
          if (resp && resp.payload) {
            this.props.fetchGraphParamSensors(this._projectId, resp.payload.parameters);
          }
        });
    }
  }

  onDelete = (event) => {
    event.preventDefault();
    const { history } = this.props;
    this.props.deleteGraph(this._projectId, this._graphId)
      .then(() => history.push(`/admin/projects/${this._projectId}/graphs`));
  };

  _genericPanel = (graph, paramSensorsPage) => (
    <Panel title="Generic options">
      <LabeledValue label="Name" value={graph.name} />
      <LabeledValue label="Description" value={graph.description} />
      <EntityInfo entity={graph} />
      <LabeledValue
        label=""
        value={(
          <PreviewLink
            type="graph"
            projectId={this._projectId}
            itemId={this._graphId}
            params={graph.parameters}
            sensorsPage={paramSensorsPage}
          />
        )}
      />
    </Panel>
  );

  _sensorsPanel = (graph) => {
    const elementsEditor = isEmpty(graph.elements)
      ? '-'
      : <GraphElementEditor elements={graph.elements} readOnly />;

    return (
      <Panel title="Metrics">
        {!isEmpty(graph.parameters) && (
          <LabeledValue
            label="Parameters"
            value={<ReadonlyTable parameters={graph.parameters} />}
          />
        )}
        <LabeledValue label="Elements" value={elementsEditor} />
      </Panel>
    );
  };

  _displayPanel = (graph) => {
    const graphMode = graph.graphMode || 'GRAPH';
    const secondaryGraphMode = graph.secondaryGraphMode || 'PIE';
    const hasYaxis = SETTINGS.checkModeHasYaxis(graphMode)
      || SETTINGS.checkModeHasYaxis(secondaryGraphMode);
    const hasHeatmap = graphMode === 'HEATMAP'
      || secondaryGraphMode === 'HEATMAP';

    const mainGraphModeTitle = getTitleByValue(graphMode, SETTINGS.GRAPH_MODE_OPTIONS);
    const secondaryGraphModeTitle = getTitleByValue(
      secondaryGraphMode,
      SETTINGS.SECONDARY_GRAPH_MODE_OPTIONS,
    );
    const stackTitle = getTitleByValue(graph.stack, SETTINGS.STACK_OPTIONS);
    const colorSchemeTitle = getTitleByValue(graph.colorScheme || 'AUTO', SETTINGS.COLOR_SCHEME_OPTIONS);
    return (
      <Panel title="Display">
        <LabeledValue label="Main graph mode" value={mainGraphModeTitle} />
        <LabeledValue label="Secondary graph mode" value={secondaryGraphModeTitle} />
        {hasHeatmap && (
          <FormGroup title="Heatmap thresholds">
            <LabeledValue label="Green" value={graph.greenValue} />
            <LabeledValue label="Yellow" value={graph.yellowValue} />
            <LabeledValue label="Red" value={graph.redValue} />
            <LabeledValue label="Violet" value={graph.violetValue} />
          </FormGroup>
        )}
        <FormGroup title="Stacking">
          <LabeledValue label="Stack" value={stackTitle} />
          <LabeledValue label="Normalize" value={graph.normalize} />
        </FormGroup>
        <FormGroup title="Color scheme">
          <LabeledValue label="Color scheme" value={colorSchemeTitle} />
          {graph.colorScheme === 'GRADIENT' && (
            <div>
              <LabeledValue label="Green" value={graph.green} />
              <LabeledValue label="Yellow" value={graph.yellow} />
              <LabeledValue label="Red" value={graph.red} />
              <LabeledValue label="Violet" value={graph.violet} />
            </div>
          )}
        </FormGroup>
        <FormGroup title="Y-axis">
          {hasYaxis && (
            <LabeledValue label="Scale" value={getTitleByValue(graph.scale, SETTINGS.SCALE_OPTIONS)} />
          )}
          <LabeledValue
            label="Number format"
            value={(<NumberFormatReadonly value={graph.numberFormat} />)}
          />
          {hasYaxis && (
            <div>
              {graph.min && <LabeledValue label="Min" value={graph.min} />}
              {graph.max && <LabeledValue label="Max" value={graph.max} />}
            </div>
          )}
        </FormGroup>
        <FormGroup title="Others">
          <LabeledValue label="Aggregation type" value={getTitleByValue(graph.aggr, SETTINGS.AGGR_OPTIONS)} />
          <LabeledValue label="Interpolate" value={getTitleByValue(graph.interpolate, SETTINGS.INTERPOLATE_OPTIONS)} />
          <LabeledValue label="Drop NaNs" value={getTitleByValue(graph.dropNans, SETTINGS.DROP_NANS_OPTIONS)} />
          <LabeledValue label="Hide no data" value={graph.hideNoData} />
          <LabeledValue label="Threshold" value={graph.threshold} />

        </FormGroup>
      </Panel>
    );
  };

  _transformPanel = (graph) => {
    const transform = graph.transform || 'NONE';
    const overLinesTransform = graph.overLinesTransform || 'NONE';
    const filter = graph.filter || 'NONE';
    const filterBy = graph.filterBy || 'AVG';
    const downsampling = graph.downsampling || 'AVG';

    return (
      <Panel title="Transformations">
        <LabeledValue label="Transform" value={getTitleByValue(transform, SETTINGS.TRANSFORM_OPTIONS)} />
        {['MOVING_AVERAGE', 'MOVING_PERCENTILE'].indexOf(overLinesTransform) >= 0 && (
          <LabeledValue label="Moving window" value={graph.movingWindow} />
        )}
        {transform === 'MOVING_PERCENTILE' && (
          <LabeledValue label="Moving percentile" value={graph.movingPercentile} />
        )}
        <LabeledValue label="Overlines transform" value={getTitleByValue(overLinesTransform, SETTINGS.OVERLINE_TRANSFORM_OPTIONS)} />
        {['PERCENTILE', 'WEIGHTED_PERCENTILE'].indexOf(overLinesTransform) >= 0 && (
          <LabeledValue label="Percentiles" value={graph.percentiles} />
        )}
        {overLinesTransform === 'WEIGHTED_PERCENTILE' && graph.bucketLabel ? (
          <LabeledValue label="Bucket label" value={graph.bucketLabel} />
        ) : null}
        <LabeledValue label="Filter" value={getTitleByValue(filter, SETTINGS.FILTER_OPTIONS)} />
        {filter !== 'NONE' && (
          <LabeledValue label="Filter by" value={getTitleByValue(filterBy, SETTINGS.AGGR_OPTIONS)} />
        )}
        {filter !== 'NONE' && (
          <LabeledValue label="Filter limit" value={graph.filterLimit} />
        )}
        <LabeledValue label="Mode" value={getTitleByValue(downsampling, SETTINGS.DOWNSAMPLING_OPTIONS)} />
        <LabeledValue label="Aggregation function" value={getTitleByValue(graph.downsamplingAggr, SETTINGS.DOWNSAMPLING_AGGR_OPTIONS)} />
        <LabeledValue label="Fill option" value={getTitleByValue(graph.downsamplingFill, SETTINGS.DOWNSAMPLING_FILL_OPTIONS)} />
        {downsampling === 'BY_INTERVAL' && (
          <LabeledValue label="Grid interval" value={graph.grid} />
        )}
        {downsampling === 'BY_POINTS' && (
          <LabeledValue label="Max points" value={graph.maxPoints} />
        )}
      </Panel>
    );
  };

  render() {
    const { graph, paramSensorsPage, projectAuth } = this.props;

    if (isEmpty(graph)) {
      return <span>Loading...</span>;
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    // Украдено из <PreviewLink />
    let link = '';
    if (isEmpty(graph.parameters)) {
      link = `/?project=${this._projectId}&graph=${this._graphId}`;
    }

    if (paramSensorsPage?.result?.length > 0) {
      const selectors = makeOldUiParamSelectors(
        graph.parameters,
        paramSensorsPage?.result[0].labels,
      );
      link = `/?project=${this._projectId}&${selectors}&graph=${this._graphId}`;
    }

    const monitoringLinkDisabled = link === '';

    link += '&__force_new=1&utm_source=solomon_graph_open_from_admin';

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Graph ${this._graphId}`} />
        {isPreOrProdIntEnv() && (
          <div className="btn-toolbar table-toolbar">
            <div className="btn-group">
              <MonitoringButton href={link} disabled={monitoringLinkDisabled} />
            </div>
          </div>
        )}
        <div className="row">
          <div className="col-xs-6">
            {this._genericPanel(graph, paramSensorsPage)}
            {this._sensorsPanel(graph)}
            <ShowActions
              editPath={`/admin/projects/${this._projectId}/graphs/${this._graphId}/edit`}
              duplicatePath={`/admin/projects/${this._projectId}/graphs/new?duplicateOf=${this._graphId}`}
              onDelete={this.onDelete}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete}
            />
          </div>
          <div className="col-xs-6">
            {this._displayPanel(graph)}
            {this._transformPanel(graph)}
          </div>
        </div>
      </div>
    );
  }
}

GraphPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  graph: PropTypes.object.isRequired,
  paramSensorsPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findGraph: PropTypes.func.isRequired,
  fetchGraphParamSensors: PropTypes.func.isRequired,
  deleteGraph: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  graph: state.graph,
  paramSensorsPage: state.graphParamSensorsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findGraph,
  deleteGraph,
  fetchGraphParamSensors,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(GraphPage);

export default wrapSubProjectReadPage(connectedPage);
