import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import TableToolbar from '../../components/TableToolbar';
import Pagination from '../../components/Pagination';

import { formatSearch, parseSearch } from '../../utils/url';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

import { findProjectGraphs } from '../../store/reducers/graphs/graphsPage';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';

const mapSearchToState = (search) => {
  const params = parseSearch(search);
  return {
    text: params.get('text') || '',
    page: params.get('page') || 0,
    pageSize: params.get('pageSize') || '',
  };
};

const mapStateToSearch = (state) => `?${formatSearch(state)}`;

const cleanState = (state) => {
  const params = {};
  const { text, page, pageSize } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }
  if (pageSize !== '' && pageSize !== 30) {
    params.pageSize = pageSize;
  }
  return params;
};

class GraphsPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.findProjectGraphs(this._projectId, cleanState(this.state));
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ ...this.state, text, page: 0 });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  doReload(newState) {
    this.setState(newState);
    const newCleanedState = cleanState(newState);
    this.props.history.replace(mapStateToSearch(newCleanedState));
    this.props.findProjectGraphs(this._projectId, newCleanedState);
  }

  render() {
    const { graphsPage, projectAuth } = this.props;
    const graphs = graphsPage.result || [];
    const pagination = graphsPage.page;
    const indexOffset = (pagination !== undefined)
      ? pagination.pageSize * pagination.current
      : 0;

    const canCreate = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);

    const newUiUrl = `https://${getMonitoringUiHost()}/projects/${this._projectId}/explorer/queries?utm_source=solomon_create_graph`;

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Graphs of ${this.props.projectName}`} />

        <TableToolbar
          createNewLink={canCreate ? `/admin/projects/${this._projectId}/graphs/new` : null}
          createNewInMonitoringLink={canCreate ? newUiUrl : null}
          filter={this.state.text}
          onFilterChange={this.onFilterChange}
        />

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>#</th>
              <th>Id</th>
              <th>Name</th>
            </tr>
          </thead>
          <tbody>
            {graphs.map((c, i) => (
              <tr key={c.id}>
                <td>{i + indexOffset + 1}</td>
                <td><Link to={`/admin/projects/${this._projectId}/graphs/${c.id}`}>{c.id}</Link></td>
                <td>{c.name}</td>
              </tr>
            ))}
          </tbody>
        </table>
        {pagination && (
          <Pagination
            activePage={pagination.current + 1}
            pageCount={pagination.pagesCount}
            pageSize={pagination.pageSize}
            totalCount={pagination.totalCount}
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        )}
      </div>
    );
  }
}

GraphsPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  graphsPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findProjectGraphs: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  graphsPage: state.graphsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findProjectGraphs,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(GraphsPage);

export default wrapSubProjectReadPage(connectedPage);
