import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import { formatSearch } from '../../utils/url';
import { makeOldUiGraphUrl } from '../../utils/SelectorUtils';
import MonitoringButton from '../old/components/ControlBar/MonitoringButton';
import { isPreOrProdIntEnv } from '../../utils/env';

class MetricsControlPanel extends PureComponent {
  makeAdminGraphUrl = () => {
    const params = {};

    if (this.props.selectors) {
      params.selectors = this.props.selectors;
    }

    if (this.props.useNewFormat) {
      params.useNewFormat = 'true';
    }

    return `/admin/projects/${this.props.projectId}/autoGraph?${formatSearch(params)}`;
  };

  render() {
    const {
      projectId, selectors, forceCluster, isRawMode, isFirstPage, metricsCount,
    } = this.props;

    const canShowGraph = metricsCount < 500;
    const canShowTable = metricsCount < 10000;

    let showGraphButton = null;
    let showGraphInMonitoringButton = null;

    if (!isFirstPage && metricsCount > 0) {
      if (canShowTable) {
        const autoGraphUrl = makeOldUiGraphUrl(projectId, selectors, forceCluster);

        if (autoGraphUrl) {
          if (canShowGraph) {
            showGraphButton = (
              <a className="btn btn-default" href={autoGraphUrl}>
                Show graph
              </a>
            );

            showGraphInMonitoringButton = isPreOrProdIntEnv() && (
              <MonitoringButton
                href={`${autoGraphUrl}&__force_new=1&utm_source=solomon_metrics_explorer`}
              />
            );
          } else {
            const tableUrl = `${autoGraphUrl}&graphMode=none&secondaryGraphMode=none`;

            showGraphButton = (
              <a className="btn btn-default" href={tableUrl}>
                Show table
              </a>
            );
          }
        }
      }
    }

    const search = {};

    if (selectors) {
      search.selectors = selectors;
    }

    if (forceCluster) {
      search.forceCluster = forceCluster;
    }

    if (this.props.useNewFormat) {
      search.useNewFormat = this.props.useNewFormat;
    }

    const formattedSearch = formatSearch(search);

    let switchModeUrl;

    if (isRawMode) {
      switchModeUrl = `/admin/projects/${projectId}/metrics?${formattedSearch}`;
    } else {
      switchModeUrl = `/admin/projects/${projectId}/rawMetrics?${formattedSearch}`;
    }

    const switchModeLink = (
      <Link className="btn btn-default" to={switchModeUrl}>
        {isRawMode ? 'Go to label values' : 'Go to raw metrics'}
      </Link>
    );

    let showGraphInAdminLink = null;

    if (canShowGraph) {
      const adminGraphUrl = this.makeAdminGraphUrl();

      if (adminGraphUrl !== '') {
        showGraphInAdminLink = (
          <Link className="btn btn-default" to={adminGraphUrl}>
            Show graph in admin
          </Link>
        );
      }
    }

    return (
      <div className="metrics-control-panel btn-toolbar">
        {showGraphInMonitoringButton}
        {showGraphButton}
        {showGraphInAdminLink}
        {switchModeLink}
      </div>
    );
  }
}

MetricsControlPanel.propTypes = {
  projectId: PropTypes.string.isRequired,
  selectors: PropTypes.string.isRequired,
  useNewFormat: PropTypes.bool.isRequired,
  forceCluster: PropTypes.string.isRequired,
  metricsCount: PropTypes.number.isRequired,
  isRawMode: PropTypes.bool.isRequired,
  isFirstPage: PropTypes.bool.isRequired,
};

export default MetricsControlPanel;
