import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabelGroup from '../../components/LabelGroups/LabelGroup';

import { formatSearch, parseSearch } from '../../utils/url';

import debounce from '../../utils/debounce';
import SelectorsControl from './SelectorsControl';
import MetricsControlPanel from './MetricsControlPanel';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import Selectors from '../../utils/Selectors';
import Selector from '../../utils/Selector';
import MetricsCountInfo from './MetricsCountInfo';
import WarningAlert from '../projects/alerts/WarningAlert';
import { mustShowClusterServiceOnly } from './mustShowClusterServiceOnly';

import { fetchLabels, clearLabels } from '../../store/reducers/sensors/sensorLabels';
import { AdminYasmAlert } from '../../components/YasmAlert';

class MetricsPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const selectors = params.get('selectors') || '';

    const forceCluster = params.get('forceCluster') || '';

    const isFirstPage = mustShowClusterServiceOnly(selectors);

    const useNewFormat = params.get('useNewFormat') || '';

    return {
      selectors, forceCluster, isFirstPage, useNewFormat, text: '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.selectors) {
      search.selectors = state.selectors;
    }

    if (state.forceCluster) {
      search.forceCluster = state.forceCluster;
    }

    if (state.useNewFormat) {
      search.useNewFormat = state.useNewFormat;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (state.selectors) {
      params.selectors = state.selectors;
    }

    if (state.forceCluster) {
      params.forceCluster = state.forceCluster;
    }

    const names = state.isFirstPage ? 'cluster,service' : '';

    if (names.length > 0) {
      params.names = names;
    }

    if (state.text) {
      params.text = state.text;
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = MetricsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.loadData();
  }

  componentDidUpdate(prevProps) {
    if (this.props.match.params.projectId !== prevProps.match.params.projectId
      || this.props.location.search !== prevProps.location.search) {
      const newState = MetricsPage.mapSearchToState(this.props.location.search);
      this.setState(newState, () => this.loadData());
    }
  }

  componentWillUnmount() {
    this.props.clearLabels();
  }

  handleSelectorsChange = (selectors) => {
    this.setState({ selectors }, () => this.doReload());
  };

  formatUrlWithNewMetricName = (_, metricName) => {
    const parsedSelectors = Selectors.parse(this.state.selectors);
    const changedSelectors = new Selectors(metricName, parsedSelectors.getSelectors());
    const changedSelectorsStr = changedSelectors.format();
    const newState = { ...this.state, selectors: changedSelectorsStr };
    return MetricsPage.mapStateToSearch(newState);
  };

  formatUrlWithNewLabelValue = (label, value) => {
    const parsedSelectors = Selectors.parse(this.state.selectors);
    const changedSelectors = parsedSelectors.addOverride(Selector.glob(label, value));
    const changedSelectorsStr = changedSelectors.format();
    const newState = { ...this.state, selectors: changedSelectorsStr };
    return MetricsPage.mapStateToSearch(newState);
  };

  handleTextChange = (event) => {
    event.preventDefault();
    this.setState({ text: event.target.value }, () => this.loadDataDebounced());
  };

  doReload = () => {
    this.props.history.push(MetricsPage.mapStateToSearch(this.state));
  };

  loadData = () => {
    const useNewFormat = this.isUseNewFormat();
    const params = MetricsPage.mapStateToParams(this.state);
    this.props.fetchLabels(this.props.match.params.projectId, params, useNewFormat);
  };

  // eslint-disable-next-line react/sort-comp
  loadDataDebounced = debounce(this.loadData, 200);

  isUseNewFormat() {
    return this.state.useNewFormat === 'true';
  }

  render() {
    const { loading, data, error } = this.props.labelsData;

    const labelsPage = data;

    const { projectId } = this.props.match.params;

    const useNewFormat = this.isUseNewFormat();

    const selectorsElement = (
      <div className="row">
        <div className="col-lg-8">
          <SelectorsControl
            selectors={this.state.selectors}
            onChange={this.handleSelectorsChange}
          />
        </div>
        {!!labelsPage && (
          <div className="col-lg-4">
            <div className="pull-right">
              <MetricsControlPanel
                projectId={projectId}
                selectors={this.state.selectors}
                useNewFormat={useNewFormat}
                forceCluster={this.state.forceCluster}
                metricsCount={labelsPage.maxCount}
                isRawMode={false}
                isFirstPage={this.state.isFirstPage}
              />
            </div>
          </div>
        )}
      </div>
    );

    let mainElement;

    if (data) {
      const { sensorNames, labels } = labelsPage;

      mainElement = (
        <div>
          <div className="btn-group" style={{ marginTop: '10px' }}>
            <input
              type="text"
              name="text"
              className="form-control"
              value={this.state.text}
              autoComplete="off"
              placeholder="Filter labels on page..."
              onChange={this.handleTextChange}
            />
          </div>
          <div>
            {!this.state.isFirstPage
              ? <MetricsCountInfo metricsCountByCluster={labelsPage.sensorsCountByCluster} />
              : <span>&nbsp;</span>}
          </div>
          {sensorNames.names.length > 0 && (
            <LabelGroup
              name="metric"
              values={sensorNames.names}
              absent={false}
              truncated={sensorNames.truncated}
              canGroupByLabel={false}
              formatUrl={this.formatUrlWithNewMetricName}
            />
          )}
          {labels.map((labelGroup) => (
            <LabelGroup
              key={labelGroup.name}
              name={labelGroup.name}
              values={labelGroup.values}
              absent={labelGroup.absent}
              truncated={labelGroup.truncated}
              formatUrl={this.formatUrlWithNewLabelValue}
            />
          ))}
        </div>
      );
    } else if (loading) {
      mainElement = <div>Loading...</div>;
    } else if (error) {
      mainElement = <WarningAlert title="Error" message={error} />;
    } else {
      mainElement = null;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Metrics of ${this.props.projectName}`} />
        <AdminYasmAlert projectId={projectId} />
        {selectorsElement}
        {mainElement}
      </div>
    );
  }
}

MetricsPage.propTypes = {
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  labelsData: PropTypes.object.isRequired,
  fetchLabels: PropTypes.func.isRequired,
  clearLabels: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  labelsData: state.sensorLabels,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  fetchLabels,
  clearLabels,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(MetricsPage),
);
