/* eslint-disable no-restricted-syntax,no-continue */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';

import { formatSearch, parseSearch } from '../../utils/url';

import SelectorsControl from './SelectorsControl';
import MetricsControlPanel from './MetricsControlPanel';
import Pagination from '../../components/Pagination';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { ADMIN } from '../../auth/Roles';
import Auth from '../../auth/Auth';
import { mustShowClusterServiceOnly } from './mustShowClusterServiceOnly';

import { clearSensors, fetchSensors } from '../../store/reducers/sensors/sensorsPage';
import { AdminYasmAlert } from '../../components/YasmAlert';

class RawMetricsPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    return {
      selectors: params.get('selectors') || '',
      page: params.get('page') || 0,
      pageSize: params.get('pageSize') || '',
      forceCluster: params.get('forceCluster') || '',
      useNewFormat: params.get('useNewFormat') || '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.selectors) {
      search.selectors = state.selectors;
    }

    if (state.forceCluster) {
      search.forceCluster = state.forceCluster;
    }

    if (state.page !== 0) {
      search.page = state.page;
    }

    if (state.pageSize && state.pageSize !== 30) {
      search.pageSize = state.pageSize;
    }

    if (state.useNewFormat) {
      search.useNewFormat = state.useNewFormat;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (state.selectors) {
      params.selectors = state.selectors;
    }

    if (state.forceCluster) {
      params.forceCluster = state.forceCluster;
    }

    if (state.page > 0) {
      params.page = state.page;
    }

    if (state.pageSize && state.pageSize !== 30) {
      params.pageSize = state.pageSize;
    }

    return params;
  }

  static formatStockpileMetricLink(name, labels, projectId) {
    const pathname = `/admin/projects/${projectId}/stockpileMetric`;
    const params = { labels };
    if (name) {
      params.name = name;
    }
    const urlParams = { pathname, search: formatSearch(params) };
    return <Link to={urlParams}><i className="glyphicon glyphicon-share-alt" /></Link>;
  }

  static formatLabelWithComma(labels) {
    const labelsArray = [];

    for (const labelKey in labels) {
      if (labelKey === 'project') {
        continue;
      }

      const labelValue = labels[labelKey];
      const label = `${labelKey}="${labelValue.replace('\\', '\\\\').replace('"', '\\"')}"`;
      labelsArray.push(label);
    }

    const labelsStr = labelsArray.join(', ');
    return labelsStr;
  }

  constructor(props) {
    super(props);
    this.state = RawMetricsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.loadData();
  }

  componentDidUpdate(prevProps) {
    if (this.props.match.params.projectId !== prevProps.match.params.projectId
      || this.props.location.search !== prevProps.location.search) {
      const newState = RawMetricsPage.mapSearchToState(this.props.location.search);
      this.setState(newState, () => this.loadData());
    }
  }

  componentWillUnmount() {
    this.props.clearSensors();
  }

  handleSelectorsChange = (selectors) => {
    this.setState({ selectors }, () => this.doReload());
  };

  handleActivePageChange = (activePage) => {
    this.setState({ page: activePage - 1 }, () => this.doReload());
  };

  handlePageSizeChange = (pageSize) => {
    this.setState({ pageSize }, () => this.doReload());
  };

  doReload = () => {
    this.props.history.replace(RawMetricsPage.mapStateToSearch(this.state));
  };

  loadData = () => {
    const params = RawMetricsPage.mapStateToParams(this.state);
    this.props.fetchSensors(this.props.match.params.projectId, params, this.isUseNewFormat());
  };

  isUseNewFormat() {
    return this.state.useNewFormat === 'true';
  }

  render() {
    const { sensorsPage } = this.props;
    const { projectId } = this.props.match.params;

    const useNewFormat = this.isUseNewFormat();

    let metricsCount;
    let mainElement;

    if (isEmpty(sensorsPage)) {
      mainElement = <div>Loading...</div>;
      metricsCount = 0;
    } else {
      const { result, page } = sensorsPage;
      metricsCount = page.totalCount;

      const canSeeStockpileMetric = Auth.hasRole(ADMIN);

      const hidePagination = !this.state.forceCluster
        || mustShowClusterServiceOnly(this.state.selectors);

      const hideTotalCount = !this.state.forceCluster;

      mainElement = (
        <div>
          <table className="table table-condensed table-hover">
            <thead>
              <tr>
                <th>Kind</th>
                {useNewFormat ? <th>Name</th> : null}
                <th>Labels</th>
                {canSeeStockpileMetric ? <th>SP link</th> : null}
                <th>Created at</th>
              </tr>
            </thead>
            <tbody>
              {result.map((metric, index) => {
                const labelsStr = RawMetricsPage.formatLabelWithComma(metric.labels);
                return (
                  // eslint-disable-next-line react/no-array-index-key
                  <tr key={index}>
                    <td>{metric.kind || ''}</td>
                    {useNewFormat ? <td>{metric.name}</td> : null}
                    <td>
                      {labelsStr}
                    </td>
                    {canSeeStockpileMetric ? (
                      <td>
                        {RawMetricsPage.formatStockpileMetricLink(
                          metric.name,
                          labelsStr,
                          projectId,
                        )}
                      </td>
                    ) : null}
                    <td>{metric.createdAt || ''}</td>
                  </tr>
                );
              })}
            </tbody>
          </table>
          <Pagination
            activePage={page.current + 1}
            pageCount={page.pagesCount}
            pageSize={page.pageSize}
            totalCount={page.totalCount}
            onActivePageChange={this.handleActivePageChange}
            onPageSizeChange={this.handlePageSizeChange}
            hidePagination={hidePagination}
            hideTotalCount={hideTotalCount}
          />
        </div>
      );
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Raw metrics of ${this.props.projectName}`} />
        <AdminYasmAlert projectId={projectId} />
        <div className="row">
          <div className="col-lg-8">
            <SelectorsControl
              selectors={this.state.selectors}
              onChange={this.handleSelectorsChange}
            />
          </div>
          <div className="col-lg-4">
            <div className="pull-right btn-toolbar">
              <MetricsControlPanel
                projectId={projectId}
                selectors={this.state.selectors}
                forceCluster={this.state.forceCluster}
                metricsCount={metricsCount}
                useNewFormat={useNewFormat}
                isRawMode
                isFirstPage={false}
              />
            </div>
          </div>
        </div>
        {mainElement}
      </div>
    );
  }
}

RawMetricsPage.propTypes = {
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  sensorsPage: PropTypes.object.isRequired,
  fetchSensors: PropTypes.func.isRequired,
  clearSensors: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  sensorsPage: state.sensorsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  fetchSensors,
  clearSensors,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(RawMetricsPage),
);
