import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import Selectors from '../../utils/Selectors';

class SelectorsControl extends PureComponent {
  static validateSelectors(selectors) {
    try {
      Selectors.parse(selectors);
      return true;
    } catch (e) {
      return false;
    }
  }

  static mapPropsToState(props) {
    return {
      selectors: props.selectors,
      isValid: SelectorsControl.validateSelectors(props.selectors),
    };
  }

  constructor(props) {
    super(props);
    this.state = SelectorsControl.mapPropsToState(props);
  }

  componentDidUpdate(prevProps) {
    if (this.props.selectors !== prevProps.selectors) {
      this.setState(SelectorsControl.mapPropsToState(this.props));
    }
  }

  onSelectorsChange = (event) => {
    event.preventDefault();
    this.setState({ selectors: event.target.value });
  };

  onChange = () => {
    const selectors = this.state.selectors;
    const isValid = SelectorsControl.validateSelectors(selectors);

    if (isValid) {
      this.props.onChange(selectors);
    }
  }

  onSelectorsKeyPress = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.onChange();
    }
  };

  render() {
    const { showButton } = this.props;

    return (
      <>
        <input
          className="form-control"
          type="text"
          name="selectors"
          value={this.state.selectors}
          placeholder="Selectors..."
          onChange={this.onSelectorsChange}
          onKeyPress={this.onSelectorsKeyPress}
          style={{ width: 360 }}
        />
        {showButton && (
          <>
            &nbsp;
            <button className="btn btn-primary" type="button" onClick={this.onChange}>Filter</button>
          </>
        )}
      </>
    );
  }
}

SelectorsControl.propTypes = {
  selectors: PropTypes.string,
  onChange: PropTypes.func.isRequired,
  showButton: PropTypes.bool,
};

SelectorsControl.defaultProps = {
  showButton: false,
  selectors: '',
};

export default SelectorsControl;
