import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabelGroup from '../../components/LabelGroups/LabelGroup';

import { mustShowClusterServiceOnly } from './mustShowClusterServiceOnly';

import { formatSearch, parseSearch } from '../../utils/url';

import debounce from '../../utils/debounce';
import SelectorsControl from './SelectorsControl';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import Selectors from '../../utils/Selectors';
import Selector from '../../utils/Selector';
import LabelValidator from '../../components/LabelGroups/LabelValidator';
import WarningAlert from '../projects/alerts/WarningAlert';

import { fetchLabels, clearLabels } from '../../store/reducers/sensors/sensorLabels';
import { AdminYasmAlert } from '../../components/YasmAlert';

class StrictValidationPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const selectors = params.get('selectors') || '';

    const forceCluster = params.get('forceCluster') || '';

    const isFirstPage = mustShowClusterServiceOnly(selectors);

    return {
      selectors, forceCluster, isFirstPage, text: '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    if (state.selectors) {
      search.selectors = state.selectors;
    }

    if (state.forceCluster) {
      search.forceCluster = state.forceCluster;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    if (!state.isFirstPage) {
      params._validationFilter = 'INVALID_ONLY';
    }

    if (state.selectors) {
      params.selectors = state.selectors;
    }

    if (state.forceCluster) {
      params.forceCluster = state.forceCluster;
    }

    if (state.isFirstPage) {
      params.names = 'cluster,service';
    }

    if (state.text) {
      params.text = state.text;
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = StrictValidationPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.loadData();
  }

  componentDidUpdate(prevProps) {
    if (this.props.match.params.projectId !== prevProps.match.params.projectId
      || this.props.location.search !== prevProps.location.search) {
      const newState = StrictValidationPage.mapSearchToState(this.props.location.search);
      this.setState(newState, () => this.loadData());
    }
  }

  componentWillUnmount() {
    this.props.clearLabels();
  }

  handleSelectorsChange = (selectors) => {
    this.setState({ selectors }, () => this.doReload());
  };

  formatUrlWithNewLabelValue = (label, value) => {
    const parsedSelectors = Selectors.parse(this.state.selectors);
    const changedSelectors = parsedSelectors.addOverride(Selector.glob(label, value));
    const changedSelectorsStr = changedSelectors.format();
    const newState = { ...this.state, selectors: changedSelectorsStr };
    return StrictValidationPage.mapStateToSearch(newState);
  };

  handleTextChange = (event) => {
    event.preventDefault();
    this.setState({ text: event.target.value }, () => this.loadDataDebounced());
  };

  doReload = () => {
    this.props.history.push(StrictValidationPage.mapStateToSearch(this.state));
  };

  loadData = () => {
    const params = StrictValidationPage.mapStateToParams(this.state);
    this.props.fetchLabels(this.props.match.params.projectId, params);
  };

  // eslint-disable-next-line react/sort-comp
  loadDataDebounced = debounce(this.loadData, 200);

  render() {
    const { loading, data, error } = this.props.labelsPage;

    const { projectId } = this.props.match.params;

    const maxSensorsCount = data ? data.maxCount : 0;

    const formattedSearch = formatSearch(this.props.location.search);

    const sensorsUrl = `/admin/projects/${projectId}/metrics?${formattedSearch}`;

    const selectorsElement = (
      <div className="row" style={{ marginBottom: '10px' }}>
        <div className="col-lg-8">
          <SelectorsControl
            selectors={this.state.selectors}
            onChange={this.handleSelectorsChange}
          />
        </div>
        <div className="col-lg-4">
          <div className="pull-right">
            <Link className="btn btn-default" to={sensorsUrl}>Go to metrics</Link>
          </div>
        </div>
      </div>
    );

    let mainElement;

    if (data) {
      const { labels } = data;

      const enableValidation = !this.state.isFirstPage;

      let invalidLabels = labels;

      if (enableValidation) {
        invalidLabels = labels.filter((label) =>
          !!LabelValidator.validateKey(label.name) || label.values.length !== 0);
      }

      if (invalidLabels.length === 0 && !this.state.isFirstPage && !this.state.text) {
        mainElement = (
          <div className="alert alert-warning">
            No invalid values found
          </div>
        );
      } else {
        mainElement = (
          <div>
            <div className="btn-group" style={{ marginTop: '10px' }}>
              <input
                type="text"
                name="text"
                className="form-control"
                value={this.state.text}
                autoComplete="off"
                placeholder="Filter labels on page..."
                onChange={this.handleTextChange}
              />
            </div>
            {invalidLabels.map((labelGroup) => (
              <LabelGroup
                key={labelGroup.name}
                name={labelGroup.name}
                values={labelGroup.values}
                absent={labelGroup.absent}
                enableValidation={enableValidation}
                truncated={labelGroup.truncated}
                formatUrl={this.formatUrlWithNewLabelValue}
              />
            ))}
          </div>
        );
      }
    } else if (loading) {
      mainElement = <div>Loading...</div>;
    } else if (error) {
      mainElement = <WarningAlert title="Error" message="Failed to load labels" />;
    } else if (maxSensorsCount === 0 && !this.state.isFirstPage) {
      mainElement = (
        <div className="alert alert-warning">
          No metrics found
        </div>
      );
    } else {
      mainElement = null;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Strict validation in ${this.props.projectName}`} />
        <AdminYasmAlert projectId={projectId} />
        <p>
          See more information about new metric labels restrictions&nbsp;
          <a
            href="https://docs.yandex-team.ru/solomon/concepts/data-model#limits"
            rel="noopener noreferrer" target="_blank"
          >
            here
          </a>
        </p>
        {selectorsElement}
        {mainElement}
      </div>
    );
  }
}

StrictValidationPage.propTypes = {
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  match: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  labelsPage: PropTypes.object.isRequired,
  fetchLabels: PropTypes.func.isRequired,
  clearLabels: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  labelsPage: state.sensorLabels,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  fetchLabels,
  clearLabels,
}, dispatch);

export default wrapSubProjectReadPage(
  connect(mapStateToProps, mapDispatchToProps)(StrictValidationPage),
);
