import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';
import { formatISO } from 'date-fns';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import MuteForm from './MuteForm';

import { parseSearch } from '../../utils/url';

import { wrapSubProjectEditPage } from '../projects/wrapSubProjectPage';

import { clearMute, findMute, saveMute } from '../../store/reducers/mutes/mutes';
import { explainMute, clearMuteExplanation, loadNextMuteExplanation } from '../../store/reducers/mutes/muteExplanation';

import { loadAlert } from '../../store/reducers/alerts/alert';

class MuteEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._muteId = props.match.params.muteId;
    this._isNew = !this._muteId;

    const parsedSearch = parseSearch(props.location.search);

    if (this._isNew) {
      this._muteId = parsedSearch.get('duplicateOf');
    }

    this._alertId = parsedSearch.get('alertId');
    this._subAlertSelectors = parsedSearch.get('selectors');
    this._subAlertSelectors = this._subAlertSelectors
      && decodeURIComponent(this._subAlertSelectors);
  }

  componentDidMount() {
    const { mute } = this.props;

    if (this._alertId) {
      this.props.loadAlert(this._projectId, this._alertId);
    }

    if (this._muteId) {
      const loadedId = mute.id;
      if (loadedId !== this._muteId) {
        this.props.findMute(this._projectId, this._muteId);
      }
    } else {
      this.props.clearMute();
    }
  }

  onSubmit = (value) => {
    const { history } = this.props;
    this.props.saveMute(this._projectId, value, this._isNew)
      .then(() => history.push(`/admin/projects/${this._projectId}/mutes/${value.id}`));
  };

  render() {
    const { mute, alert } = this.props;
    const title = this._isNew ? 'New Mute' : `Mute ${this._muteId}`;
    const isLoading = (this._muteId && isEmpty(mute)) || (this._alertId && isEmpty(alert));

    const editingMute = { ...mute };

    if (this._isNew && !isEmpty(alert)) {
      editingMute.from = formatISO(new Date());
      editingMute.type = editingMute.type || { selectors: {} };
      editingMute.type.selectors.alertSelector = `alert=${alert.id}`;
      if (this._subAlertSelectors) {
        editingMute.type.selectors.labelSelectors = this._subAlertSelectors;
      }
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <MuteForm
            projectId={this._projectId}
            mute={editingMute}
            isNew={this._isNew}
            onSubmit={this.onSubmit}
            explainMute={this.props.explainMute}
            loadNextMuteExplanation={this.props.loadNextMuteExplanation}
            muteExplanation={this.props.muteExplanation}
            clearMuteExplanation={this.props.clearMuteExplanation}
          />
          )}
      </div>
    );
  }
}

MuteEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  mute: PropTypes.object.isRequired,
  alert: PropTypes.object,
  loadAlert: PropTypes.func.isRequired,
  muteExplanation: PropTypes.object,
  findMute: PropTypes.func.isRequired,
  saveMute: PropTypes.func.isRequired,
  clearMute: PropTypes.func.isRequired,
  explainMute: PropTypes.func.isRequired,
  loadNextMuteExplanation: PropTypes.func.isRequired,
  clearMuteExplanation: PropTypes.func.isRequired,
};

MuteEditPage.defaultProps = {
  alert: null,
  muteExplanation: null,
};

const mapStateToProps = (state) => ({
  mute: state.mute,
  alert: state.alert,
  muteExplanation: state.muteExplanation,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findMute,
  saveMute,
  clearMute,
  loadAlert,
  explainMute,
  loadNextMuteExplanation,
  clearMuteExplanation,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(MuteEditPage);

export default wrapSubProjectEditPage(connectedPage);
