import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import ApiCallExample from '../../components/ApiCallExample';
import { inputTargetValue } from '../../utils/inputs';
import FormDate from '../../components/forms/FormDate';
import RelatedAlertsTable from '../alerts/components/RelatedAlerts/RelatedAlertsTable';
import Panel from '../../components/Panel';

class MuteForm extends PureComponent {
  static muteToState = (mute, isNew) => {
    if (mute.id) {
      const state = { ...mute };
      state.id = isNew ? '' : state.id;
      return state;
    }
    return { ...mute, id: '' };
  };

  static mapPropsToState = (mute, isNew) => ({
    mute: MuteForm.muteToState(mute, isNew),
    apiVisibility: true,
  });

  static updateMuteType(mute, type, update) {
    const muteType = mute.type || {};
    const muteTypeContent = muteType[type] || {};
    const newMuteTypeContent = { ...muteTypeContent, ...update };
    const newMuteType = { ...muteType, [type]: newMuteTypeContent };
    return { ...mute, type: newMuteType };
  }

  constructor(props) {
    super(props);

    this.state = MuteForm.mapPropsToState(props.mute, props.isNew);
    this._isNewMute = props.isNew;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    // We cannot use setState() because it performs a shallow merge of
    // previous state with the new one. But here we need entirely update state.
    if (this.props.projectId !== prevProps.projectId
      || (!this.props.isNew && this.props.mute?.id !== prevProps.mute?.id)) {
      this.state = MuteForm.mapPropsToState(this.props.mute, this.props.isNew);
    }
  }

  onInputChange = (event) => {
    const { target, value: valueFromWrapper } = event;
    const value = valueFromWrapper ?? inputTargetValue(target);
    const change = {};
    change[target.name] = value;

    this.changeMute(change);
  };

  onRowsChange = (rows) => {
    this.changeMute({ rows });
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.getCleanMute());
  };

  onExplain = (event) => {
    event.preventDefault();
    if (this.state.showExplain) {
      this.setState({ showExplain: false }, () => {
        this.setState({ showExplain: true });
      });
    } else {
      this.setState({ showExplain: true });
    }
  }

  onJsonStateChange = (mute) => {
    this.setState({ mute });
  };

  onApiVisibilityChange = (apiVisibility) => {
    this.setState({ apiVisibility });
  };

  changeMute = (update) => {
    this.setState({ mute: { ...this.state.mute, ...update } });
  };

  onSelectorsInputChange = (event) => {
    const { target } = event;
    const value = inputTargetValue(target);
    const change = {};
    switch (target.name) {
      case 'alertSelector':
        change.alertSelector = `alert${value}`;
        break;
      default:
        change[target.name] = value;
    }

    this.setState({ mute: MuteForm.updateMuteType(this.state.mute, 'selectors', change) });
  };

  getCleanMute = () => {
    const mute = { ...this.state.mute };

    delete mute.createdAt;
    delete mute.updatedAt;
    delete mute.status;
    if (this._isNewMute) {
      delete mute.version;
    }

    return mute;
  }

  loadExplain = (_, __, params) => {
    this.props.explainMute(this.props.projectId, this.state.mute, params);
  }

  render() {
    const { projectId } = this.props;

    const { mute, apiVisibility } = this.state;

    const {
      id, description, ticketId, from, to, type, name,
    } = mute;

    const muteType = type || {};
    const muteSelectors = muteType.selectors || {};
    let alertSelector = muteSelectors.alertSelector ?? '';
    if (alertSelector.startsWith('alert')) {
      alertSelector = alertSelector.substring(5);
    }

    return (
      <div className="form-horizontal">
        <div className={apiVisibility ? 'col-lg-6 col-md-7' : 'col-lg-10 col-md-11'}>
          <FormInput
            type="text" name="id" label="Id" value={id} onChange={this.onInputChange}
            help="Unique mute identificator"
            disabled={!this._isNewMute}
          />
          <FormInput
            type="text" name="name" label="Name" value={name} onChange={this.onInputChange}
            help="Mute name"
          />
          <FormInput
            type="text" name="description" label="Description" value={description} onChange={this.onInputChange}
            help="Details about the mute"
          />
          <FormInput
            type="text" name="ticketId" label="Ticket" value={ticketId} onChange={this.onInputChange}
            help="Related Startrek ticket"
          />
          <FormDate
            type="text" name="from" label="From" value={from} onChange={this.onInputChange}
            help="Beginning time of the mute"
          />
          <FormDate
            type="text" name="to" label="To" value={to} onChange={this.onInputChange}
            help="Ending time of the mute"
            dateReference={from}
            shortcuts={['30m', '1h', '1d', '1w', 'Until Monday', 'Until tomorrow']}
          />
          <FormInput
            type="text" name="alertSelector" label="Alert selector" value={alertSelector} onChange={this.onSelectorsInputChange}
            prefix="alert"
            placeholder="=uptime-*-alert"
            help="Selector for muting alert ids"
          />
          <FormInput
            type="text" name="labelSelectors" label="Subalert labels selector" value={muteSelectors.labelSelectors} onChange={this.onSelectorsInputChange}
            help="Selectors for muting subalerts by labels"
          />
          {this.state.showExplain
            && (
              <Panel title="Affected alerts">
                <RelatedAlertsTable
                  projectId={this.props.projectId}
                  alertsPage={this.props.muteExplanation.data || {}}
                  clearAlerts={this.props.clearMuteExplanation}
                  filters={false}
                  loadAlerts={this.loadExplain}
                  relationId={this.state.mute.id}
                  loadNextAlerts={this.loadExplain}
                  notificationStats={false}
                  warnEmpty
                  emptyMessage="No affected alerts found."
                  /**
                   * Temporary disabled until figure out how to display
                   * affected subalerts muted stats
                   * */
                  mutedStats={false}
                />
              </Panel>
            )}
          <FormButtons onSubmit={this.onSubmit} onExplain={this.onExplain} />
        </div>
        <div className={apiVisibility ? 'col-lg-6 col-md-5' : 'col-lg-2 col-md-1'}>
          <ApiCallExample
            path={`/projects/${projectId}/mutes`}
            value={this.getCleanMute()}
            isNew={this._isNewMute}
            visible={apiVisibility}
            onChange={this.onJsonStateChange}
            onChangeVisibility={this.onApiVisibilityChange}
          />
        </div>
      </div>
    );
  }
}

MuteForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  mute: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
  explainMute: PropTypes.func.isRequired,
  // loadNextMuteExplanation: PropTypes.func.isRequired,
  muteExplanation: PropTypes.object,
  clearMuteExplanation: PropTypes.func.isRequired,
};

MuteForm.defaultProps = {
  muteExplanation: null,
};

export default MuteForm;
