import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import LabeledValue from '../../components/LabeledValue';
import Panel from '../../components/Panel';
import ShowActions from '../../components/ShowActions';

import { deleteMute, findMute, MUTE_FOUND } from '../../store/reducers/mutes/mutes';

import EntityInfo from '../../components/EntityInfo';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';
import { formatLocalDate } from '../../utils/localDate';
import { explainMute } from '../../store/reducers/mutes/muteExplanation';
import { loadMuteAlerts, loadNextMuteAlerts, clearMuteAlerts } from '../../store/reducers/mutes/muteAlertsPage';
import RelatedAlertsTable from '../alerts/components/RelatedAlerts/RelatedAlertsTable';
import TicketLink from '../../components/TicketLink/TicketLink';
import MuteStatus from './components/MuteStatus';

function SelectorsMuteProps(props) {
  const { selectors } = props;

  function isTrivial(labelSelectors) {
    return labelSelectors === '' || labelSelectors === '{}';
  }

  const labelSelectorsContent = !isTrivial(selectors.labelSelectors) ? (
    <pre style={{ whiteSpace: 'pre-wrap' }}>
      {selectors.labelSelectors}
    </pre>
  ) : null;

  return (
    <div>
      {selectors.alertSelector
      && <LabeledValue label="Alert selector" value={(<pre>{selectors.alertSelector}</pre>)} />}
      {!isTrivial(selectors.labelSelectors)
      && <LabeledValue label="Label selector" value={labelSelectorsContent} />}
    </div>
  );
}

SelectorsMuteProps.propTypes = {
  selectors: PropTypes.object.isRequired,
};

class MutePage extends PureComponent {
  constructor(props) {
    super(props);
    const { projectId, muteId } = props.match.params;
    this._projectId = projectId;
    this._muteId = muteId;
  }

  componentDidMount() {
    this.props.findMute(this._projectId, this._muteId)
      .then((action) => {
        if (action.type === MUTE_FOUND) {
          this.props.loadMuteAlerts(this._projectId, action.payload.id);
        }
      });
  }

  componentDidUpdate(prevProps) {
    const { params } = this.props.match;
    const prevParams = prevProps.match.params;
    if (params.id !== prevParams.id) {
      this._projectId = params.projectId;
      this._muteId = params.id;
    }
  }

  onDelete = (event) => {
    event.preventDefault();
    const { history } = this.props;
    this.props.deleteMute(this._projectId, this._muteId)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${this._projectId}/mutes`);
        }
      });
  };

  render() {
    const {
      mute,
      projectAuth,
      muteAlertsPage,
      // eslint-disable-next-line
      loadMuteAlerts, loadNextMuteAlerts, clearMuteAlerts,
    } = this.props;

    if (isEmpty(mute)) {
      return <div>Loading...</div>;
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    if (isEmpty(mute) || isEmpty(muteAlertsPage)) {
      return <span>Loading...</span>;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Mute ${this._muteId}`} />
        <div className="row">
          <div className="col-lg-8">
            <Panel title="Generic options">
              <LabeledValue label="From" value={formatLocalDate(new Date(mute.from))} />
              <LabeledValue label="To" value={formatLocalDate(new Date(mute.to))} />
              {mute.type.selectors
              && <SelectorsMuteProps projectId={this._projectId} selectors={mute.type.selectors} />}
              {mute.description
              && <LabeledValue label="Description" value={mute.description} />}
              <LabeledValue label="Status" value={<MuteStatus mute={mute} />} />
              {mute.ticketId && <LabeledValue label="Ticket" value={<TicketLink text={mute.ticketId} />} />}
              <EntityInfo entity={mute} />
            </Panel>
            <ShowActions
              editPath={`/admin/projects/${this._projectId}/mutes/${this._muteId}/edit`}
              duplicatePath={`/admin/projects/${this._projectId}/mutes/new?duplicateOf=${this._muteId}`}
              onDelete={this.onDelete}
              canEdit={canEdit}
              canChangeState={false}
              canDelete={canDelete}
            />
          </div>
          <div className="col-lg-4">
            {muteAlertsPage && (
              <Panel title="Affected alerts">
                <RelatedAlertsTable
                  relationId={this._muteId}
                  projectId={this._projectId}
                  alertsPage={muteAlertsPage}
                  loadAlerts={loadMuteAlerts}
                  loadNextAlerts={loadNextMuteAlerts}
                  clearAlerts={clearMuteAlerts}
                  notificationStats={false}
                  warnEmpty
                  emptyMessage="No affected alerts found"
                  /**
                   * Temporary disabled until figure out how to display
                   * affected subalerts muted stats
                   * */
                  mutedStats={false}
                />
              </Panel>
            )}
          </div>
        </div>
      </div>
    );
  }
}

MutePage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  mute: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  findMute: PropTypes.func.isRequired,
  loadMuteAlerts: PropTypes.func.isRequired,
  loadNextMuteAlerts: PropTypes.func.isRequired,
  clearMuteAlerts: PropTypes.func.isRequired,
  muteAlertsPage: PropTypes.object.isRequired,
  deleteMute: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  mute: state.mute,
  muteExplanation: state.muteExplanation,
  muteAlertsPage: state.muteAlertsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findMute,
  deleteMute,
  explainMute,
  loadMuteAlerts,
  loadNextMuteAlerts,
  clearMuteAlerts,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(MutePage);

export default wrapSubProjectReadPage(connectedPage);
