import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import TicketLink from '../../components/TicketLink/TicketLink';

import { formatLocalDate } from '../../utils/localDate';
import { formatSearch, parseSearch } from '../../utils/url';
import { wrapSubProjectReadPage } from '../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

import MuteStatus from './components/MuteStatus';
import { loadProjectMutes, loadNextProjectMutes } from '../../store/reducers/mutes/mutesPage';
import { MUTE_STATUS_VALUES } from './constants';
import LoadMore from '../../components/LoadMore/LoadMore';

class MutesPage extends PureComponent {
  static mapSearchToState = (search) => {
    const parsedSearch = parseSearch(search);
    const nameFilter = parsedSearch.get('name') || '';
    const statusFilter = parsedSearch.get('status') || 'ALL';
    const pageSize = parsedSearch.get('pageSize') || '30';

    return {
      nameFilter,
      statusFilter,
      pageSize,
    };
  };

  static mapStateToParams = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.filterByName = state.nameFilter;
    }
    if (state.statusFilter !== 'ALL') {
      params.filterByStates = [state.statusFilter];
    }
    if (state.pageSize) {
      params.pageSize = state.pageSize;
    } else {
      params.pageSize = '30';
    }
    return params;
  };

  static mapStateToSearch = (state) => {
    const params = {};
    if (state.nameFilter) {
      params.name = state.nameFilter;
    }
    if (state.statusFilter !== 'ALL') {
      params.status = state.statusFilter;
    }
    if (state.pageSize !== '30') {
      params.pageSize = state.pageSize;
    }
    return `?${formatSearch(params)}`;
  };

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this.state = MutesPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.props.loadProjectMutes(this._projectId, MutesPage.mapStateToParams(this.state));
  }

  componentDidUpdate(prevProps) {
    const { projectId } = this.props.match.params;
    const prevProjectId = prevProps.match.params.projectId;
    if (projectId !== prevProjectId
      || this.props.location.search !== prevProps.location.search) {
      this._projectId = projectId;

      const newState = MutesPage.mapSearchToState(this.props.location.search);
      this.setState(newState);
      this.props.loadProjectMutes(this._projectId, MutesPage.mapStateToParams(newState));
    }
  }

  onMuteFilterChange = (event) => {
    event.preventDefault();
    this.setState({ [event.target.name]: event.target.value }, () => this.reloadMutes());
  }

  onLoadMore = () => {
    const pageToken = this.props.mutesPage.nextPageToken;
    const newParams = { ...MutesPage.mapStateToParams(this.state), pageToken, pageSize: 30 };
    this.props.loadNextProjectMutes(this._projectId, newParams);
  };

  onActivePageChange = (activePage) => {
    this.doReload({ ...this.state, page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ ...this.state, pageSize, page: 0 });
  };

  reloadMutes() {
    const search = MutesPage.mapStateToSearch(this.state);
    this.props.history.replace(search);
  }

  render() {
    const { mutesPage, projectAuth } = this.props;
    const mutes = mutesPage.items || [];
    const canCreate = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Mutes of ${this.props.projectName}`} />

        {canCreate && (
          <div className="btn-toolbar table-toolbar">
            <Link className="btn btn-success btn-outline" to={`/admin/projects/${this._projectId}/mutes/new`}>
              <i className="glyphicon glyphicon-plus" />
              {' '}
              Create new
            </Link>
          </div>
        )}

        <table className="table table-condensed table-hover">
          <thead>
            <tr>
              <th>Name</th>
              <th>Ticket</th>
              <th>Status</th>
              <th>From</th>
              <th>To</th>
            </tr>
          </thead>
          <tbody>
            <tr>
              <td>
                <input
                  className="form-control"
                  name="nameFilter"
                  value={this.state.nameFilter}
                  placeholder="Type to filter mute names"
                  onChange={this.onMuteFilterChange}
                  ref={(el) => { this.inputEl = el; }}
                />
              </td>
              <td />
              <td>
                <select
                  className="form-control"
                  name="statusFilter"
                  value={this.state.statusFilter}
                  onChange={this.onMuteFilterChange}
                >
                  <option value="ALL">Any state</option>
                  {MUTE_STATUS_VALUES.map((option) => (
                    <option key={option.value} value={option.value}>
                      {option.title}
                    </option>
                  ))}
                </select>
              </td>
              <td colSpan={2} />
            </tr>
            {mutes.map((mute) => ((
              <tr key={mute.id}>
                <td><Link to={`/admin/projects/${this._projectId}/mutes/${mute.id}`}>{mute.name || mute.id}</Link></td>
                <td>
                  <TicketLink text={mute.ticketId} />
                </td>
                <td>
                  <MuteStatus mute={mute} />
                </td>
                <td>
                  {formatLocalDate(new Date(mute.from))}
                </td>
                <td>
                  {formatLocalDate(new Date(mute.to))}
                </td>
              </tr>
            )))}
          </tbody>
        </table>
        {mutesPage.nextPageToken ? <LoadMore onLoadMore={this.onLoadMore} /> : null}
      </div>
    );
  }
}

MutesPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  mutesPage: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  loadProjectMutes: PropTypes.func.isRequired,
  loadNextProjectMutes: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  mutesPage: state.mutesPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadProjectMutes,
  loadNextProjectMutes,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(MutesPage);

export default wrapSubProjectReadPage(connectedPage);
