/* eslint-disable no-continue,import/no-cycle */
import React, { Component } from 'react';
import PropTypes from 'prop-types';

import UserLinksBasic from '../../../utils/UserLinksBasic';
import UrlUtils from '../../../utils/url/UrlUtils';
import DashboardWithSpansUtils from './utils/DashboardWithSpansUtils';
import { CellState } from './utils/TableWithSpansBuilder';
import OldDashboardPanel from './OldDashboardPanel';
import InternalOrExternalLink from '../../../components/InternalOrExternalLink';

import './style.css';

const MIN_DASHBOARD_ELEMENT_HEIGHT = 170;

const FORWARDED_PARAMS = [
  UserLinksBasic.LEGEND_QUERY_ARG,
  UserLinksBasic.AUTOREFRESH_QA,
  UserLinksBasic.NEED_POINTS_PARAM,
  UserLinksBasic.SHOW_CENSORED_INTERVALS,
];

const STATES = {
  INIT: 0,
  MOUNT_DASHBOARD_SCHEMA: 1,
  MOUNT_ELEMENTS: 2,
};

/**
 *
 * @param {LinkedHashMap} paramValues
 * @param {UrlParsed} parsedDashUrl
 */
function fillUrlByForceCluster(paramValues, parsedDashUrl) {
  const forceCluster = parsedDashUrl.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_CLUSTER_PARAM)
   || parsedDashUrl.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_HOST_PARAM);
  if (forceCluster) {
    paramValues.put(UserLinksBasic.FORCE_CLUSTER_PARAM, forceCluster);
  }
}

function fillUrlByForwardedParams(graphUrl, parsedDashUrl) {
  const paramValues = parsedDashUrl.getLogicalQueryArgsMapByNames(FORWARDED_PARAMS);
  fillUrlByForceCluster(paramValues, parsedDashUrl);
  return UrlUtils.fillDefaultsMap(graphUrl, paramValues);
}

function fillUrlByTimeParams(graphUrl, parsedDashUrl) {
  const parsedGraphUrl = UrlUtils.parseUrl(graphUrl);
  const graphB = parsedGraphUrl.getLogicalQueryArgOrEmpty(UserLinksBasic.B_QA);
  const graphE = parsedGraphUrl.getLogicalQueryArgOrEmpty(UserLinksBasic.E_QA);

  if (!graphB && !graphE) {
    const timeParamValues = parsedDashUrl.getLogicalQueryArgsMapByNames(
      [UserLinksBasic.B_QA, UserLinksBasic.E_QA],
    );
    return UrlUtils.updateParameters(graphUrl, timeParamValues);
  }

  return graphUrl;
}

function fixUrlForDashboard(graphUrl, dashboardUrl) {
  let fixedGraphUrl = UserLinksBasic.fixSolomonUrl(graphUrl);

  if (UserLinksBasic.isGeneric(graphUrl)) {
    const parsedDashUrl = UrlUtils.parseUrl(dashboardUrl);
    fixedGraphUrl = fillUrlByForwardedParams(graphUrl, parsedDashUrl);
    return fillUrlByTimeParams(fixedGraphUrl, parsedDashUrl);
  }

  return graphUrl;
}

function calculateDivMaxHeight($app, $div, isLocal) {
  if (!isLocal) {
    const divTop = $div.offsetTop;
    const divBottom = $app.scrollHeight - divTop - $div.scrollHeight;
    return window.innerHeight - divTop - divBottom;
  }

  return $app.innerHeight;
}

function getLongVerticalDivHeight($app, $div, heightMultiplier, isLocal) {
  let divHeight = calculateDivMaxHeight($app, $div, isLocal) * heightMultiplier;
  if (divHeight < 100) {
    divHeight = 100;
  }

  return divHeight;
}

class OldDashboard extends Component {
  constructor(props) {
    super(props);
    this.state = { init: STATES.INIT };
  }

  componentDidMount() {
    // Hack to re-render dashboard after placeholder element mounting
    // eslint-disable-next-line react/no-did-mount-set-state
    this.setState({ init: STATES.MOUNT_DASHBOARD_SCHEMA });
  }

  componentDidUpdate(prevProps) {
    if (prevProps.dashboard !== this.props.dashboard) {
      this.setState({ init: STATES.INIT });
    } else if (this.state.init === STATES.INIT) {
      this.setState({ init: STATES.MOUNT_DASHBOARD_SCHEMA });
    } else if (this.state.init === STATES.MOUNT_DASHBOARD_SCHEMA) {
      this.setState({ init: STATES.MOUNT_ELEMENTS });
    }
  }

  renderDashboardBody() {
    const {
      url, appElement, isLocal, dashboard,
    } = this.props;

    const dashboardUrl = url;

    const $div = this._tablePlaceholder;

    const table = DashboardWithSpansUtils.makeTableFromDashboardRows(dashboard.rows);
    const tableHeight = getLongVerticalDivHeight(appElement, $div, dashboard.height, isLocal);
    const tableWidth = $div.clientWidth;

    if (table.getRowCount() === 0) {
      return null;
    }

    const relativeRowHeight = table.getRowCount() > 0 ? 100 / table.getRowCount() : 0;

    const rowHeight = Math.max(MIN_DASHBOARD_ELEMENT_HEIGHT, tableHeight / table.getRowCount());

    const rowElements = [];

    for (let i = 0; i < table.getRowCount(); ++i) {
      const rowCellElements = [];

      for (let j = 0; j < table.getColumnCount(); ++j) {
        const cell = table.getCellOrEmpty(i, j);
        const item = cell.getContentOrNull();

        if (cell.getState() === CellState.INSIDE_SPANNED_CELL) {
          continue;
        }

        const cellHeight = rowHeight * cell.getRowspan();
        const cellWidth = Math.trunc((tableWidth * cell.getColspan()) / table.getColumnCount());

        const cellClassName = `dashboard-element${cell.isEmptyCell() ? ' dashboard-empty-element' : ''}`;
        const cellStyle = { height: `${cellHeight}px`, width: `${cellWidth}px` };

        const itemId = item && item.uuid ? item.uuid : `panel_${i}_${j}`;

        if (!item) {
          rowCellElements.push((
            <td
              key={itemId}
              className={cellClassName}
              colSpan={cell.getColspan()}
              rowSpan={cell.getRowspan()}
              style={cellStyle}
            />
          ));
          continue;
        }

        const itemUrl = fixUrlForDashboard(item.url, dashboardUrl);
        const fixedUrl = fixUrlForDashboard(item.rawUrl, dashboardUrl);
        const rawItemUrl = UserLinksBasic.fixOldAdminUrl(fixedUrl);

        const body = this.state.init === STATES.MOUNT_ELEMENTS ? (
          <OldDashboardPanel id={itemId} item={item} url={itemUrl} />
        ) : null;

        const cellElement = (
          <td
            key={itemId}
            className={cellClassName}
            colSpan={cell.getColspan()}
            rowSpan={cell.getRowspan()}
            style={cellStyle}
          >
            <div className="dashboard-element-head">
              <span className="dashboard-element-title">
                <InternalOrExternalLink href={rawItemUrl}>
                  {item.title}
                </InternalOrExternalLink>
              </span>
              <small className="dashboard-element-subtitle">
                {item.subtitle}
              </small>
            </div>
            {body}
          </td>
        );

        rowCellElements.push(cellElement);
      }

      const rowElement = (
        <tr
          key={`dashboard-row-${i}`}
          className="dashboard-row"
          style={{ height: `${relativeRowHeight}%` }}
        >
          {rowCellElements}
        </tr>
      );

      rowElements.push(rowElement);
    }

    return (
      <table className="dashboard" style={{ height: `${tableHeight}px` }}>
        <tbody>
          {rowElements}
        </tbody>
      </table>
    );
  }

  render() {
    let body = null;

    if (this._tablePlaceholder && this.state.init !== STATES.INIT) {
      body = this.renderDashboardBody();
    }

    return (
      <div
        className="dashboard-placeholder"
        ref={(el) => { this._tablePlaceholder = el; }}
      >
        {body}
      </div>
    );
  }
}

OldDashboard.propTypes = {
  url: PropTypes.string.isRequired,
  appElement: PropTypes.instanceOf(Element).isRequired,
  isLocal: PropTypes.bool.isRequired,
  dashboard: PropTypes.object.isRequired,
};

export default OldDashboard;
