/* eslint-disable max-len,no-param-reassign,max-classes-per-file */
export const CellState = {
  EMPTY: 'EMPTY',
  FULL: 'FULL',
  INSIDE_SPANNED_CELL: 'INSIDE_SPANNED_CELL',
};

export class Cell {
  constructor(contentOrNull, state, rowspan, colspan) {
    this._contentOrNull = contentOrNull;
    this._state = state;
    this._rowspan = rowspan;
    this._colspan = colspan;
  }

  static empty() {
    return new Cell(null, CellState.EMPTY, 1, 1);
  }

  getState() {
    return this._state;
  }

  getContentOrNull() {
    return this._contentOrNull;
  }

  getRowspan() {
    return this._rowspan;
  }

  getColspan() {
    return this._colspan;
  }

  isEmptyCell() {
    return this._state !== CellState.FULL;
  }
}

export class TableWithSpansBuilder {
  constructor() {
    this._rows = [];
    this._columnCount = 0;
  }

  getCellOrEmpty(rowIndex, columnIndex) {
    if (rowIndex >= 0 && rowIndex < this._rows.length) {
      const row = this._rows[rowIndex];
      if (columnIndex >= 0 && columnIndex < row.length) {
        return row[columnIndex];
      }
    }
    return Cell.empty();
  }

  setSpannedCell(rowIndex, columnIndex, rowspan, colspan, content) {
    colspan = Math.max(1, colspan);
    rowspan = Math.max(1, rowspan);

    for (let curColumnIndex = columnIndex; curColumnIndex < columnIndex + colspan; ++curColumnIndex) {
      const cell = this.getCellOrEmpty(rowIndex, curColumnIndex);
      if (cell.getState() !== CellState.EMPTY) {
        columnIndex = curColumnIndex + 1;
      }
    }

    for (let curRowIndex = rowIndex; curRowIndex < rowIndex + rowspan; ++curRowIndex) {
      for (let curColumnIndex = columnIndex; curColumnIndex < columnIndex + colspan; ++curColumnIndex) {
        this.setCell(
          curRowIndex, curColumnIndex,
          null, CellState.INSIDE_SPANNED_CELL,
          rowspan, colspan,
        );
      }
    }

    this.setCell(rowIndex, columnIndex, content, CellState.FULL, rowspan, colspan);
  }

  setCell(rowIndex, columnIndex, content, cellState, rowspan, colspan) {
    while (rowIndex >= this._rows.length) {
      this._rows.push([]);
    }

    const row = this._rows[rowIndex];

    while (columnIndex >= row.length) {
      row.push(Cell.empty());
    }

    row[columnIndex] = new Cell(content, cellState, rowspan, colspan);

    if (columnIndex >= this._columnCount) {
      this._columnCount = columnIndex + 1;

      this._rows.forEach((curRow) => {
        while (curRow.length < this._columnCount) {
          curRow.push(Cell.empty());
        }
      });
    }
  }

  getColumnCount() {
    return this._columnCount;
  }

  getRowCount() {
    return this._rows.length;
  }
}

export default TableWithSpansBuilder;
