import React, { PureComponent } from 'react';
import ReactDOM from 'react-dom';
import PropTypes from 'prop-types';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import GraphModeLines from '../../../solomon-graph/lines/GraphModeLines';
import { isTrue } from '../../../utils/boolean';
import toDouble from '../../../solomon-graph/common/toDouble';
import GraphModeHeatmap from '../../../solomon-graph/heatmap/GraphModeHeatmap';
import GraphModeBars from '../../../solomon-graph/bars/GraphModeBars';
import GraphModePie from '../../../solomon-graph/pie/GraphModePie';
import ShardHealthAlert from './ShardHealthAlert/ShardHealthAlert';
import GraphErrorPopup from './GraphErrorPopup/GraphErrorPopup';

function createLines(graphConf, graphDataWithMeta) {
  if (isTrue(graphConf.hideNoData)) {
    return graphDataWithMeta.filter((seriess) => seriess.salmonStats.hasData);
  }
  return graphDataWithMeta;
}

export function lineDataToAggrValues(lines, sensorLink) {
  return lines.map((line) => ({
    color: line.color,
    label: line.label,
    value: toDouble(line.aggregatedValue),
    // stateful
    link: sensorLink ? sensorLink.addCurrentPageParams(line.link) : line.link,
    show: line.show,
    cookie: '',
  }));
}

function createAggrValues(graphConf, lines, sensorLink) {
  const fixedLines = createLines(graphConf, lines);
  return lineDataToAggrValues(fixedLines, sensorLink);
}

class GraphModeEngine extends PureComponent {
  componentDidMount() {
    this._mountWarningAlert();
  }

  componentDidUpdate() {
    this._mountWarningAlert();
  }

  componentWillUnmount() {
    if (this.$alertWarningPlaceholder) {
      ReactDOM.unmountComponentAtNode(this.$alertWarningPlaceholder);
    }
  }

  // It's dirty hack from old implementation to render inside chart area
  _mountWarningAlert() {
    if (!this.props.primary || !this.$graph || !this.$alertWarningPlaceholder) {
      return;
    }

    const top = 0;
    let left = 5;

    const $graphSvg = this.$graph.querySelector('.chart-root-div > svg > svg:nth-child(1)');
    if ($graphSvg) {
      left += parseInt($graphSvg.getAttribute('x'), 10);
    }

    const healthAlert = (
      <ShardHealthAlert health={this.props.graphData.health} top={top} left={left} />
    );

    ReactDOM.render(healthAlert, this.$alertWarningPlaceholder);
  }

  render() {
    const {
      url, graphMode, graphData, lines, graphConf, graphFormatter, sensorLink,
    } = this.props;

    let graph = null;

    if (graphMode === UserLinksBasic.GRAPH_MODE_GRAPH) {
      const fixedLines = createLines(graphConf, lines);

      graph = (
        <GraphModeLines
          url={url}
          graphData={graphData}
          lines={fixedLines}
          graphConf={graphConf}
          graphFormatter={graphFormatter}
          width={this.props.width}
          graphOnly={this.props.graphOnly}
          isLocal={this.props.isLocal}
        />
      );
    } else if (graphMode === UserLinksBasic.GRAPH_MODE_NONE) {
      if (this.props.primary) {
        graph = (<div style={{ textAlign: 'center' }}>Graph mode is &quot;None&quot;</div>);
      } else {
        graph = null;
      }
    } else {
      const aggrValues = createAggrValues(graphConf, lines, sensorLink);

      if (graphMode === UserLinksBasic.GRAPH_MODE_BARS) {
        graph = (
          <GraphModeBars
            graphFormatter={graphFormatter}
            aggrValues={aggrValues}
            graphConf={graphConf}
            distributionMode={false}
          />
        );
      } else if (graphMode === UserLinksBasic.GRAPH_MODE_DISTRIBUTION) {
        graph = (
          <GraphModeBars
            graphFormatter={graphFormatter}
            aggrValues={aggrValues}
            graphConf={graphConf}
            distributionMode
          />
        );
      } else if (graphMode === UserLinksBasic.GRAPH_MODE_PIE_CHART) {
        graph = (
          <GraphModePie
            aggrValues={aggrValues}
            graphFormatter={graphFormatter}
          />
        );
      } else if (graphMode === UserLinksBasic.GRAPH_MODE_HEATMAP) {
        graph = (
          <GraphModeHeatmap
            aggrValues={aggrValues}
            graphConf={graphConf}
            graphFormatter={graphFormatter}
          />
        );
      } else {
        throw new Error(`unknown graph mode: ${graphMode}`);
      }
    }

    let graphErrorPopup = null;

    if (this.props.graphError) {
      graphErrorPopup = (
        <GraphErrorPopup
          message="Failed to load graph"
          error={this.props.graphError}
        />
      );
    }

    return (
      <div
        className="graph" style={{ position: 'relative', height: this.props.height }}
        ref={(el) => { this.$graph = el; }}
      >
        {graph}
        <div
          className="alert-warning-placeholder"
          ref={(el) => { this.$alertWarningPlaceholder = el; }}
        />
        {graphErrorPopup}
      </div>
    );
  }
}

GraphModeEngine.propTypes = {
  url: PropTypes.string.isRequired,
  graphMode: PropTypes.string.isRequired,
  graphData: PropTypes.object.isRequired,
  graphError: PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
  graphFormatter: PropTypes.object.isRequired,
  lines: PropTypes.array.isRequired,
  graphConf: PropTypes.object.isRequired,
  sensorLink: PropTypes.object.isRequired,
  height: PropTypes.number.isRequired,
  width: PropTypes.number.isRequired,
  graphOnly: PropTypes.bool.isRequired,
  isLocal: PropTypes.bool.isRequired,
  primary: PropTypes.bool,
};

GraphModeEngine.defaultProps = {
  graphError: null,
  primary: false,
};

export default GraphModeEngine;
