import { formatRgb, getGradientColor } from '../../../utils/color/colors';
import { CssColorName } from '../../../utils/color/CssColorName';
import { parseColorOrNull } from '../../../utils/color/CssColor';

const DEFAULT_WARNING_GT_THRESHOLD = 0;
const DEFAULT_DANGER_GT_THRESHOLD = 25;

const WARNING_THRESHOLD_KEY = 'warningHealthThreshold';
const DANGER_THRESHOLD_KEY = 'dangerHealthThreshold';
const ALERT_HEATH_MUTED_KEY = 'alertHeathMuted';

const WARNING_COLOR = parseColorOrNull(CssColorName.ORANGE);
const DANGER_COLOR = parseColorOrNull(CssColorName.RED);

class HealthAlerting {
  static OK = 0;

  static WARNING = 1;

  static DANGER = 2;

  static getDefaultWarningGtThreshold() {
    return HealthAlerting.getPropertyOrDefault(WARNING_THRESHOLD_KEY, DEFAULT_WARNING_GT_THRESHOLD);
  }

  static getDefaultDangerGtThreshold() {
    return HealthAlerting.getPropertyOrDefault(DANGER_THRESHOLD_KEY, DEFAULT_DANGER_GT_THRESHOLD);
  }

  static isMuted() {
    return window.localStorage.getItem(ALERT_HEATH_MUTED_KEY) === 'true';
  }

  static getPropertyOrDefault(key, defaultValue) {
    const propertyStr = window.localStorage.getItem(key);
    const result = parseInt(propertyStr, 10);
    if (isNaN(result)) {
      return defaultValue;
    }

    return result;
  }

  static computeStatus(value) {
    if (HealthAlerting.isMuted() || value <= HealthAlerting.getDefaultWarningGtThreshold()) {
      return HealthAlerting.OK;
    }
    if (value <= HealthAlerting.getDefaultDangerGtThreshold()) {
      return HealthAlerting.WARNING;
    }
    return HealthAlerting.DANGER;
  }

  static computeBorderColor(value) {
    if (HealthAlerting.isMuted()) {
      return '';
    }

    const warningGtThreshold = HealthAlerting.getDefaultWarningGtThreshold();
    const dangerGtThreshold = HealthAlerting.getDefaultDangerGtThreshold();

    if (value <= warningGtThreshold) {
      return '';
    }
    if (value <= dangerGtThreshold) {
      const pos = (value - warningGtThreshold) / (dangerGtThreshold - warningGtThreshold);
      return getGradientColor(pos, WARNING_COLOR, DANGER_COLOR);
    }

    return formatRgb(DANGER_COLOR);
  }
}

export default HealthAlerting;
