import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import entries from 'lodash/entries';
import Modal from '../../../../components/Modal';
import UrlUtils from '../../../../utils/url/UrlUtils';
import UserLinksBasic from '../../../../utils/UserLinksBasic';
import BrowserUtils from '../../../../utils/BrowserUtils';
import FRONTEND_VERSION from '../../../../utils/frontendVersion';

const IDDQD = 'iddqd';

function createAdminStockpileSensorUrl(sensorLabelsMap) {
  const { project } = sensorLabelsMap;

  if (!project) {
    return '';
  }

  const formattedLabels = entries(sensorLabelsMap)
    .filter((entry) => UserLinksBasic.PROJECT !== entry[0])
    .map((entry) => {
      const key = entry[0];
      const value = entry[1];

      const formattedValue = `"${value.replace(/"/g, '\\"')}"`;

      return `${key}=${formattedValue}`;
    })
    .join(', ');

  const b = BrowserUtils.getLogicalQueryArgOrEmpty('b');
  const e = BrowserUtils.getLogicalQueryArgOrEmpty('e');

  const pathname = `/admin/projects/${project}/stockpileMetric`;
  return UrlUtils.addThreeParameters(
    pathname,
    'labels', formattedLabels,
    'b', b,
    'e', e,
  );
}

class IddqdModal extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { pos: 0, modalShown: false };
  }

  componentDidMount() {
    document.addEventListener('keypress', this.onDocumentKeypress);
  }

  componentWillUnmount() {
    document.removeEventListener('keypress', this.onDocumentKeypress);
  }

  onDocumentKeypress = (event) => {
    if (this.state.modalShown) {
      return;
    }

    if (IDDQD.charAt(this.state.pos) === event.key) {
      if (this.state.pos === IDDQD.length - 1) {
        this.setState({ pos: 0, modalShown: true });
      } else {
        this.setState({ pos: this.state.pos + 1 });
      }
    } else {
      this.setState({ pos: 0 });
    }
  };

  onCancel = () => {
    this.setState({ modalShown: false });
  };

  linesWithDebugInfo() {
    const rows = [];
    const { lines } = this.props;

    if (lines) {
      for (let i = 0; i < lines.length; ++i) {
        const line = lines[i];
        const { label } = line;
        const sensorLabelsMap = line.sensorLabels || {};

        const adminStockpileSensorHref = createAdminStockpileSensorUrl(sensorLabelsMap);

        rows.push((
          <tr key={label}>
            <td>{label}</td>
            <td>
              {!!adminStockpileSensorHref && (
                <a href={adminStockpileSensorHref}>Go to admin</a>
              )}
            </td>
          </tr>
        ));
      }
    }

    return (
      <>
        <h4>Lines with debug info</h4>
        <div style={{ maxHeight: '200px', overflow: 'scroll' }}>
          <table
            className="table table-condensed"
            style={{ width: '100%', fontFamily: 'monospace', whiteSpace: 'nowrap' }}
          >
            <thead>
              <tr>
                <th>Label</th>
                <th>SP metric</th>
              </tr>
            </thead>
            <tbody>
              {rows}
            </tbody>
          </table>
        </div>
      </>
    );
  }

  render() {
    if (!this.state.modalShown) {
      return null;
    }

    return (
      <Modal title="God mode" onCancel={this.onCancel} isOpen>
        <div>
          <div className="iddqd-lines-diagnostic-placeholder">
            <div>
              Version:
              {FRONTEND_VERSION}
            </div>
          </div>
          {this.linesWithDebugInfo()}
        </div>
      </Modal>
    );
  }
}

IddqdModal.propTypes = {
  lines: PropTypes.array.isRequired,
};

export default IddqdModal;
