/* eslint-disable guard-for-in,no-restricted-syntax */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';

import Modal from '../../../components/Modal';
import { ControlKind } from './utils/GraphSetting';
import FormSelect from '../../../components/forms/FormSelect';
import FormInput from '../../../components/forms/FormInput';
import FormArea from '../../../components/forms/FormArea';
import NumberFormatSelector from '../../graphs/NumberFormatSelector';
import FormElement from '../../../components/forms/FormElement';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import { parseSearch } from '../../../utils/url';
import PROJECT_CONFIGS from '../utils/projectConfigs';
import parseBoolean from '../../../utils/boolean';
import BrowserUtils from '../../../utils/BrowserUtils';

const LABEL_WIDTH = 6;

function buildUrl(settings, defaultConf, newConf) {
  const searchParams = parseSearch(window.location.search);

  for (const setting of settings) {
    const value = setting.getFromOrDefault(newConf);
    if (setting.isDefaultValue(value, defaultConf) || !setting.isShow(newConf)) {
      searchParams.delete(setting.queryArg);
    } else {
      searchParams.set(setting.queryArg, value);
    }
  }

  return `?${searchParams.toString()}`;
}

class OldGraphConfigureModal extends PureComponent {
  static mapConfToState(conf) {
    const fixedConf = {};

    for (const name in conf) {
      const value = conf[name];

      if (!(value === null || value === undefined)) {
        fixedConf[name] = `${value}`;
      }
    }

    return fixedConf;
  }

  constructor(props) {
    super(props);
    this.state = { conf: OldGraphConfigureModal.mapConfToState(props.conf) };
  }

  componentDidUpdate(prevProps) {
    if (this.props.isOpen !== prevProps.isOpen || !isEqual(this.props.conf, prevProps.conf)) {
      this.setState({ conf: OldGraphConfigureModal.mapConfToState(this.props.conf) });
    }
  }

  onSelectChange = (event) => {
    event.preventDefault();
    this.setState({ conf: { ...this.state.conf, [event.target.name]: event.target.value } });
  };

  onBoolSelectChange = (event) => {
    const { name, value } = event.target;
    this.setState({ conf: { ...this.state.conf, [name]: value } });
  };

  onTextChange = (event) => {
    event.preventDefault();
    this.setState({ conf: { ...this.state.conf, [event.target.name]: event.target.value } });
  };

  onNumberFormatChange = (value) => {
    this.setState({ conf: { ...this.state.conf, [UserLinksBasic.GRAPH_NUMBER_FORMAT]: value } });
  };

  onApply = (event) => {
    event.preventDefault();
    const url = buildUrl(this.props.settings, this.props.defaultConf, this.state.conf);
    this.props.onApply(url);
  };

  onCancel = (event) => {
    event.preventDefault();
    this.props.onCancel();
  };

  renderSetting(setting, conf, onOffSuffix) {
    const value = setting.getRawOrEmpty(conf);

    switch (setting.kind) {
      case ControlKind.DROPDOWN:
        return (
          <FormSelect
            key={setting.queryArg}
            label={setting.title}
            labelWidth={LABEL_WIDTH}
            name={setting.queryArg}
            options={setting.options}
            value={value}
            defaultValue={setting.defaultValue}
            onChange={this.onSelectChange}
          />
        );
      case ControlKind.BOOLEAN_DROPDOWN: {
        const fieldOptions = [...setting.options];
        if (setting.queryArg === UserLinksBasic.STACK_PARAM) {
          fieldOptions[0].title = `Default (${onOffSuffix})`;
        }

        const boolValue = parseBoolean(value);

        const strValue = boolValue === null ? '' : `${boolValue}`;

        return (
          <FormSelect
            key={setting.queryArg}
            label={setting.title}
            labelWidth={LABEL_WIDTH}
            name={setting.queryArg}
            options={fieldOptions}
            value={strValue}
            defaultValue={setting.defaultValue}
            onChange={this.onBoolSelectChange}
          />
        );
      }
      case ControlKind.TEXTFIELD:
        return (
          <FormInput
            key={setting.queryArg}
            type="text"
            name={setting.queryArg}
            label={setting.title}
            labelWidth={LABEL_WIDTH}
            value={value}
            placeholder={setting.hint}
            onChange={this.onTextChange}
          />
        );
      case ControlKind.MULTILINETEXT:
        return (
          <FormArea
            key={setting.queryArg}
            name={setting.queryArg}
            label={setting.title}
            labelWidth={LABEL_WIDTH}
            value={value}
            onChange={this.onTextChange}
          />
        );
      case ControlKind.NUMBER_FORMAT:
        return (
          <FormElement label="Number format" labelWidth={LABEL_WIDTH} key={setting.queryArg}>
            <NumberFormatSelector
              value={value}
              onChange={this.onNumberFormatChange}
            />
          </FormElement>
        );
      default:
        throw new Error(`unknown option kind: ${setting.kind}`);
    }
  }

  render() {
    const { conf } = this.state;
    const { projectId } = this.props;
    const projectConfig = PROJECT_CONFIGS[projectId] || {};
    const onOffSuffix = projectConfig.stack !== false ? 'On' : 'Off';

    const isAutoGraphParam = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM)
      === UserLinksBasic.GRAPH_VALUE_AUTO;

    const options = this.props.settings
      .filter((setting) => setting.isShow(conf)
        && setting.queryArg !== UserLinksBasic.IGNORE_MIN_STEP_MILLIS)
      .filter((setting) => (isAutoGraphParam || setting.queryArg !== UserLinksBasic.STACK_PARAM))
      .map((setting) => this.renderSetting(setting, conf, onOffSuffix));

    return (
      <Modal
        title="Graph config"
        okTitle="Apply"
        isOpen={this.props.isOpen}
        onOk={this.onApply}
        onCancel={this.onCancel}
      >
        <div className="form-horizontal">
          {options}
        </div>
      </Modal>
    );
  }
}

OldGraphConfigureModal.propTypes = {
  projectId: PropTypes.string.isRequired,
  conf: PropTypes.object.isRequired,
  defaultConf: PropTypes.object.isRequired,
  settings: PropTypes.array.isRequired,
  isOpen: PropTypes.bool.isRequired,
  onApply: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

export default OldGraphConfigureModal;
