/* eslint-disable operator-linebreak */
/* eslint-disable no-param-reassign */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import OldPageTitle from '../../../components/OldPageTitle';
import OldHeader from '../../../components/Header/OldHeader';
import TopLevelAlerts from '../../../components/TopLevelAlerts';

import OldControlBar from '../components/ControlBar/OldControlBar';
import OldGraphWithLegend from './OldGraphWithLegend';
import ErrorBlock from '../components/ErrorBlock';
import OldSelectors from '../components/OldSelectors/OldSelectors';

import IddqdModal from './IddqdModal/IddqdModal';
import TopLevelGraphAlerts from '../components/TopLevelGraphAlerts';
import TitleWithDescription from '../components/TitleWithDescription/TitleWithDescription';
import AutorefreshUtils from './utils/AutorefreshUtils';

import UserLinksBasic from '../../../utils/UserLinksBasic';
import UrlUtils from '../../../utils/url/UrlUtils';
import GraphSettings from './utils/GraphSettings';
import BrowserUtils from '../../../utils/BrowserUtils';
import YasmAlert from '../../../components/YasmAlert';

import {
  loadGraph, reloadGraph, pinGraph, unpinGraph, clearGraph,
} from '../../../store/reducers/old/oldGraphs';

import './style.css';
import redirectToNewUi from '../utils/redirectToNewUi';
import { isPreOrProdIntEnv } from '../../../utils/env';
import { isTrue } from '../../../utils/boolean';
import {
  createConfiguredGraphOrDashboardUrl,
  createExplorerUrl,
} from '../utils/graphToExpressionTransformations';
import NewUiAlert from '../../../components/NewUiAlert';
import { loadUserSettings } from '../../../store/reducers/others/userSettings';

function isText(value) {
  if (!value) {
    return false;
  }
  try {
    return JSON.parse(value).queryViewMode === 'text';
  } catch (e) {
    console.error(e);
    return false;
  }
}

class OldGraphPage extends PureComponent {
  componentDidMount() {
    this.props.loadUserSettings();
    if (!this.props.isLocal && this.props.graphOnly) {
      document.body.className = 'solomon-body-fullscreen';
    }
    if (this.props.canRedirectToNewUi) {
      redirectToNewUi(isText(this.props.uiSettings));
    } else {
      this.fetchData();
    }
  }

  componentDidUpdate(prevProps) {
    if (!this.props.canRedirectToNewUi) {
      if (this.props.isLocal) {
        this.fetchIfNeededForLocal(prevProps);
      } else {
        this.fetchIfNeededForGlobal(prevProps);
      }
    }
    if (!this.props.isLocal && this.props.graphOnly) {
      document.body.className = 'solomon-body-fullscreen';
    } else {
      document.body.className = '';
    }
  }

  componentWillUnmount() {
    document.body.className = '';
    this.clearCoro();
    this.props.clearGraph(this.props.id);
  }

  onPinGraph = (pinProps) => {
    this.props.pinGraph(this.props.id, pinProps);
  };

  onUnpinGraph = (pinId) => {
    this.props.unpinGraph(this.props.id, pinId);
  };

  setCoro(interval) {
    this._coro = setTimeout(() => this.reloadData(), interval);
  }

  clearCoro() {
    if (this._coro) {
      clearTimeout(this._coro);
    }
  }

  fetchIfNeededForGlobal(prevProps) {
    if (this.props.location !== prevProps.location) {
      this.fetchData();
    }
  }

  fetchIfNeededForLocal(prevProps) {
    if (this.props.id !== prevProps.id) {
      this.fetchData();
    }
  }

  runAutorefreshIfNeeded() {
    const parsedUrl = UrlUtils.parseUrl(this.props.url);

    if (AutorefreshUtils.isAutorefresh(parsedUrl)) {
      const interval = AutorefreshUtils.autorefreshIntervalMillis(parsedUrl);
      this.setCoro(interval);
    }
  }

  computeBaseParamsForGeneric() {
    const baseParams = UrlUtils.parseUrlQueryArgs(this.props.url);
    const width = this.props.isLocal ? this.props.appElement.scrollWidth : window.innerWidth;
    const urlParsed = UrlUtils.parseUrl(this.props.url);
    // eslint-disable-next-line max-len
    let points = urlParsed.getLogicalQueryArgNonNegativeIntegerOr(UserLinksBasic.NEED_POINTS_PARAM, 0);
    if (points === 0) {
      points = Math.trunc(width / 2);
    }

    baseParams.put(UserLinksBasic.NEED_POINTS_PARAM, `${points}`);

    return baseParams.toObject();
  }

  fetchData() {
    this.clearCoro();

    const baseParams = this.computeBaseParamsForGeneric();

    this.props.loadGraph(this.props.id, baseParams).then(() => this.runAutorefreshIfNeeded());
  }

  reloadData() {
    this.clearCoro();

    const baseParams = this.computeBaseParamsForGeneric();

    try {
      this.props.reloadGraph(this.props.id, baseParams)
        .catch((e) => { console.error('failed to reload graph', e); })
        .then(() => this.runAutorefreshIfNeeded());
    } catch (e) {
      console.error('failed to reload graph', e);
      this.runAutorefreshIfNeeded();
    }
  }

  render() {
    const {
      url, appElement, isLocal, graphOnly, canRedirectToNewUi, uiSettings,
    } = this.props;
    const urlParsed = UrlUtils.parseUrl(url);

    if (canRedirectToNewUi) {
      return 'Redirect to new UI...';
    }

    const graphId = urlParsed.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM);
    let title;

    if (graphId === 'auto') {
      title = 'Graph';
    } else {
      title = `Graph ${graphId}`;
    }

    const { oldGenericData, oldGraphData, oldGraphError } = this.props;

    let mainElement;
    let menu = {};

    const genericResult = oldGenericData.data;
    const graphDataResult = oldGraphData.data;
    if (oldGenericData.error != null) {
      mainElement = (
        <ErrorBlock error={oldGenericData.error} />
      );
    } else if (oldGenericData.data === null) {
      if (oldGenericData.loading) {
        mainElement = (<div className={`${graphOnly ? 'center' : ''}`}>Loading...</div>);
      } else {
        mainElement = null;
      }
    } else {
      title = genericResult.windowTitle;
      menu = genericResult.menu ? genericResult.menu : {};

      const { fullItem } = genericResult;

      let controlBar = null;
      let graphWithLegend = null;
      let iddqdModal = null;

      const topLevelGraphAlerts = [];

      if (genericResult && genericResult.noncriticalError) {
        const alert = { ...genericResult.noncriticalError, id: 'metadata_error' };

        if (alert.text && alert.text.indexOf('Solomon will only display truncated result') >= 0) {
          alert.alarm = true;
          alert.text = 'More than 10000 metrics found, Solomon will only display truncated result';
        }

        topLevelGraphAlerts.push(alert);
      } else if (graphDataResult && graphDataResult.noncriticalError) {
        const alert = { ...graphDataResult.noncriticalError, id: 'data_error' };
        topLevelGraphAlerts.push(alert);
      }

      if (graphDataResult && graphDataResult.criticalErrors) {
        topLevelGraphAlerts.push(...graphDataResult.criticalErrors);
      }

      let isYasmProject = false;
      if (BrowserUtils.getLogicalQueryArgOrEmpty('project').startsWith('yasm_')) {
        isYasmProject = true;
      } else if (genericResult && genericResult.fullItem && genericResult.fullItem.dataProjectId.startsWith('yasm_')) {
        isYasmProject = true;
      }

      const yasmAlert = isYasmProject ? <YasmAlert /> : null;

      const noncriticalAlert = (
        <>
          <TopLevelGraphAlerts alerts={topLevelGraphAlerts} />
          {yasmAlert}
        </>
      );

      const isAutoGraph = !(fullItem && fullItem.graph);
      let newUiUrl = '';

      if (fullItem) {
        const conf = fullItem.graphConf;

        // eslint-disable-next-line no-nested-ternary,max-len
        const defaultConf = graphDataResult && graphDataResult.graphConf ? graphDataResult.graphConf : {};

        if (!graphOnly && isPreOrProdIntEnv()) {
          if (fullItem.graph) {
            const genId = fullItem.graph.generatedId;
            if (genId) {
              newUiUrl = createConfiguredGraphOrDashboardUrl(genId);
            }
          } else {
            newUiUrl = createExplorerUrl(isText(uiSettings));
          }

          newUiUrl += '&utm_source=solomon_graph_view';
        }

        controlBar = (
          <OldControlBar
            genericResult={genericResult}
            settings={GraphSettings.settings}
            newUiUrl={newUiUrl}
            conf={conf}
            defaultConf={defaultConf}
            pinPage={this.onPinGraph}
            unpinPage={this.onUnpinGraph}
          />
        );

        if (!graphDataResult || graphDataResult.hideGraph !== true) {
          graphWithLegend = (
            <OldGraphWithLegend
              url={url}
              appElement={appElement}
              graphMetaData={fullItem}
              graphData={oldGraphData}
              graphError={oldGraphError}
              isLocal={isLocal}
              graphOnly={graphOnly}
            />
          );
        }

        if (oldGraphData.data) {
          const lines = oldGraphData.data.graphData;
          iddqdModal = (
            <IddqdModal lines={lines} />
          );
        }
      }

      if (graphOnly) {
        mainElement = (
          <>
            {noncriticalAlert}
            {graphWithLegend}
          </>
        );
      } else {
        const description = fullItem ? fullItem.description : '';

        mainElement = (
          <div>
            <TopLevelAlerts admin={false} />
            <NewUiAlert type={isAutoGraph ? 'autoGraph' : 'graph'} url={newUiUrl} />
            <OldSelectors selectors={genericResult.selectors} />
            <TitleWithDescription
              title={genericResult.graphTitle}
              description={description}
            />
            {controlBar}
            {noncriticalAlert}
            {graphWithLegend}
            {iddqdModal}
          </div>
        );
      }
    }

    if (graphOnly) {
      return (
        <div className={`${isLocal ? '' : 'container-fluid '}container-solomon container-solomon-fullscreen`}>
          {mainElement}
        </div>
      );
    }

    return (
      <div>
        <OldPageTitle title={title} />
        <OldHeader menu={menu} />
        <div className="container-fluid container-solomon">
          {mainElement}
        </div>
      </div>
    );
  }
}

OldGraphPage.propTypes = {
  id: PropTypes.string.isRequired,
  url: PropTypes.string.isRequired,
  appElement: PropTypes.instanceOf(Element).isRequired,
  isLocal: PropTypes.bool.isRequired,
  graphOnly: PropTypes.bool.isRequired,
  location: PropTypes.object,
  canRedirectToNewUi: PropTypes.bool.isRequired,
  oldGenericData: PropTypes.object.isRequired,
  oldGraphData: PropTypes.object.isRequired,
  oldGraphError: PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
  loadGraph: PropTypes.func.isRequired,
  reloadGraph: PropTypes.func.isRequired,
  pinGraph: PropTypes.func.isRequired,
  unpinGraph: PropTypes.func.isRequired,
  clearGraph: PropTypes.func.isRequired,
  loadUserSettings: PropTypes.func.isRequired,
  uiSettings: PropTypes.string,
};

OldGraphPage.defaultProps = {
  oldGraphError: null,
  location: null,
  uiSettings: '',
};

const defaultInsideState = {
  loading: false,
  data: null,
  error: null,
};

const defaultOldGraphState = {
  oldGenericData: defaultInsideState,
  oldGraphData: defaultInsideState,
  oldGraphError: null,
};

const mapStateToProps = (state, ownProps) => {
  const oldGraph = state.oldGraphs[ownProps.id];

  const forceOld = isTrue(BrowserUtils.getLogicalQueryArgOrEmpty('__old'));
  const forceNew = isTrue(
    BrowserUtils.getLogicalQueryArgOrEmpty('__force_new'),
  );

  const notForbiddenToRedirect =
    !ownProps.graphOnly &&
    state.userSettings.data.settings.redirectToNewUi === 'true' &&
    !forceOld;

  const canRedirectToNewUi =
    isPreOrProdIntEnv() && (notForbiddenToRedirect || forceNew);

  if (oldGraph) {
    return {
      oldGenericData: oldGraph.generic,
      oldGraphData: oldGraph.graphData,
      oldGraphError: oldGraph.error,
      uiSettings: state.userSettings.data.settings.ui,
      canRedirectToNewUi,
    };
  }
  const uiSettings = state.userSettings.data.settings.ui;
  return { ...defaultOldGraphState, canRedirectToNewUi, uiSettings };
};

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadGraph,
  reloadGraph,
  pinGraph,
  unpinGraph,
  clearGraph,
  loadUserSettings,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(OldGraphPage);
