import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import GraphModeEngine from './GraphModeEngine';
import GraphConfUtils from './utils/GraphConfUtils';
import GraphFormatter from '../../data/GraphFormatter';
import LegendStyle from './OldLegend/LegendStyle';
import DisconnectIcon from './DisconnectIcon/DisconnectIcon';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import SensorLink from './utils/SensorLink';
import OldImprovedLegend from './OldLegend/OldImprovedLegend';

const MIN_GRAPH_HEIGHT = 100;

const NONE = 'NONE';
const NEED_TO_RENDER_GRAPH = 'NEED_TO_RENDER_GRAPH';

function calculateInnerElementHeight(placeholder, selector) {
  const innerElement = placeholder.querySelector(selector);
  if (innerElement) {
    return innerElement.clientHeight;
  }

  return 0;
}

class OldGraphWithLegend extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { status: NONE };
  }

  componentDidUpdate(prevProps) {
    if (this.props.graphData.loading) {
      this.setState({ status: NONE });
    } else {
      const freshlyLoaded = prevProps.graphData.loading && !this.props.graphData.loading
        && !this.props.graphData.error
        && this.props.graphData.data;

      if (this.props.graphOnly && this.state.status === NONE && freshlyLoaded) {
        this.setState({ status: NEED_TO_RENDER_GRAPH });
      }
    }
  }

  calculateGraphHeight(graphMode, graphOnly, isLocal) {
    let graphHeight;
    if (graphMode === UserLinksBasic.GRAPH_MODE_NONE) {
      graphHeight = 50;
    } else if (graphOnly) {
      const totalHeight = isLocal ? this.props.appElement.clientHeight : window.innerHeight;

      const legendHeight = calculateInnerElementHeight(this.props.appElement, '.legend');

      let noncriticalHeight = calculateInnerElementHeight(this.props.appElement, '.noncriticalAlert-placeholder');

      // Check that ".alert" element also has 20px bottom margin
      if (noncriticalHeight > 20) {
        noncriticalHeight += 20;
      }

      graphHeight = totalHeight - legendHeight - noncriticalHeight - 10;

      if (graphHeight <= MIN_GRAPH_HEIGHT) {
        graphHeight = MIN_GRAPH_HEIGHT;
      }
    } else {
      graphHeight = 400;
    }
    return graphHeight;
  }

  render() {
    const {
      url, graphMetaData, graphData, graphError, isLocal, graphOnly,
    } = this.props;

    const r = GraphConfUtils.fromQueryArgs(url);

    const graphConf = GraphConfUtils.merge(graphMetaData.graphConf || {}, r);

    const graphMode = GraphConfUtils.getGraphMode(graphConf, true);
    const secondaryGraphMode = GraphConfUtils.getGraphMode(graphConf, false);

    let graph = null;
    let legendAndSecondaryGraph = null;

    if (graphData.loading) {
      graph = (
        <div className="graph-data-loading" style={{ textAlign: 'center' }}>
          Loading data...
        </div>
      );
    } else if (graphData.error) {
      const graphHeight = this.calculateGraphHeight(graphMode, graphOnly, isLocal);
      const showIcon = graphMode !== 'none';

      graph = (
        <div className="graph" style={{ position: 'relative', height: graphHeight }}>
          <DisconnectIcon showIcon={showIcon} error={graphData.error} />
        </div>
      );
    } else if (!graphData.data) {
      graph = null;
    } else {
      const graphWidth = this.props.appElement.clientWidth;
      const graphHeight = this.calculateGraphHeight(graphMode, graphOnly, isLocal);

      const graphDataResult = graphData.data;

      const lines = graphDataResult.graphData;

      const graphFormatter = GraphFormatter.makeFromGraphData(graphConf, lines, graphHeight);

      const legendStyle = LegendStyle.fromUrl(url, graphOnly);

      const sensorLink = SensorLink.fromUrl(url);

      const legend = (
        <OldImprovedLegend
          style={legendStyle}
          rows={lines}
          columns={graphDataResult.columns}
          formatter={graphFormatter}
          sensorLink={sensorLink}
        />
      );

      if (this.props.debug || !graphOnly || this.state.status === NEED_TO_RENDER_GRAPH) {
        graph = (
          <GraphModeEngine
            url={url}
            graphMode={graphMode}
            graphConf={graphConf}
            graphData={graphDataResult}
            graphError={graphError}
            graphFormatter={graphFormatter}
            lines={lines}
            sensorLink={sensorLink}
            height={graphHeight}
            width={graphWidth}
            graphOnly={graphOnly}
            isLocal={isLocal}
            primary
          />
        );
      }

      if (!graphOnly) {
        const secondaryGraph = (
          <div className="secondary-graph-placeholder">
            <GraphModeEngine
              url={url}
              graphMode={secondaryGraphMode}
              graphData={graphDataResult}
              graphFormatter={graphFormatter}
              lines={lines}
              graphConf={graphConf}
              sensorLink={sensorLink}
              height={155}
              width={0}
              graphOnly={graphOnly}
              isLocal={isLocal}
            />
          </div>
        );

        legendAndSecondaryGraph = (
          <div className="legend-and-secondary-graph">
            <div className="legend-and-secondary-graph-row">
              {secondaryGraph}
              <div className="legend">
                {legend}
              </div>
            </div>
          </div>
        );
      } else {
        legendAndSecondaryGraph = (
          <div className="legend-and-secondary-graph">
            <div className="legend">
              {legend}
            </div>
          </div>
        );
      }
    }

    return (
      <div className="graph-with-legend">
        {graph}
        {legendAndSecondaryGraph}
      </div>
    );
  }
}

OldGraphWithLegend.propTypes = {
  url: PropTypes.string.isRequired,
  appElement: PropTypes.instanceOf(Element).isRequired,
  graphMetaData: PropTypes.object.isRequired,
  graphData: PropTypes.object.isRequired,
  graphError: PropTypes.oneOfType([PropTypes.object, PropTypes.string]),
  isLocal: PropTypes.bool.isRequired,
  graphOnly: PropTypes.bool.isRequired,
  debug: PropTypes.bool,
};

OldGraphWithLegend.defaultProps = {
  graphError: null,
  debug: false,
};

export default OldGraphWithLegend;
