/* eslint-disable jsx-a11y/label-has-for,react/no-array-index-key */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import compareTextWithNumber from '../../../../utils/sortTextWithNumber';
import SensorsChecksSet from './SensorsChecksSet';
import UserLinksBasic from '../../../../utils/UserLinksBasic';
import InternalOrExternalLink from '../../../../components/InternalOrExternalLink';

const SORT_DIRECTION = {
  UNSORTED: 'unsorted',
  ASC: 'asc',
  DESC: 'desc',
};

const COLUMN_TYPE = {
  LABEL: 'label',
  NUMBER: 'number',
};

class LegendSortedTable extends PureComponent {
  static _sortRows(rows, columnType, columnId, sortDirection) {
    if (sortDirection === SORT_DIRECTION.UNSORTED) {
      return rows;
    }

    const sortMultiplier = sortDirection === SORT_DIRECTION.ASC ? 1 : -1;

    if (columnType === COLUMN_TYPE.LABEL) {
      const columnIndex = parseInt(columnId, 10);

      const sortedRows = [...rows];

      sortedRows.sort((a, b) => {
        const aName = a.getLabelColumnValueOrNull(columnIndex);
        const bName = b.getLabelColumnValueOrNull(columnIndex);

        return sortMultiplier * compareTextWithNumber(aName, bName);
      });

      return sortedRows;
    }

    if (columnType === COLUMN_TYPE.NUMBER) {
      const sortedRows = [...rows];

      sortedRows.sort((a, b) => {
        const aValue = a[columnId];
        const bValue = b[columnId];

        if (isNaN(aValue) && isNaN(bValue)) {
          return 0;
        } if (isNaN(aValue)) {
          return 1;
        } if (isNaN(bValue)) {
          return -1;
        }

        return sortMultiplier * (aValue - bValue);
      });

      return sortedRows;
    }

    throw new Error(`unknown column type: ${columnType}`);
  }

  static _setSelectionRange(newRowChecks, sortedRows, checkedId, lastCheckedId) {
    let commonChecked;

    if (lastCheckedId in newRowChecks) {
      commonChecked = newRowChecks[lastCheckedId];
    } else {
      const lastCheckedRows = sortedRows.filter((r) => r.salmonId === lastCheckedId);

      if (lastCheckedRows.length > 0) {
        commonChecked = lastCheckedRows[0].checked;
      } else {
        return;
      }
    }

    let initSelection = false;

    for (let i = 0; i < sortedRows.length; ++i) {
      const row = sortedRows[i];
      const rowId = row.salmonId;

      if (rowId === lastCheckedId || rowId === checkedId) {
        // eslint-disable-next-line no-param-reassign
        newRowChecks[rowId] = commonChecked;
        if (initSelection) {
          break;
        } else {
          initSelection = true;
        }
      } else if (initSelection) {
        // eslint-disable-next-line no-param-reassign
        newRowChecks[rowId] = commonChecked;
      }
    }
  }

  constructor(props) {
    super(props);

    this.state = {
      allChecked: props.rows.some((row) => row.checked),
      lastCheckedId: '',
      rowChecks: {},
      sortedColumnType: '',
      sortedColumnId: '',
      sortDirection: SORT_DIRECTION.UNSORTED,
      sortedRows: props.rows,
    };
  }

  onColumnClick = (event) => {
    let thElement = event.target;

    while (!thElement || thElement.tagName !== 'TH') {
      thElement = thElement.parentElement;
    }

    const { columnId, columnType, sortDirection } = thElement.dataset;

    let newSortDirection;

    if (sortDirection === SORT_DIRECTION.UNSORTED) {
      newSortDirection = SORT_DIRECTION.DESC;
    } else if (sortDirection === SORT_DIRECTION.DESC) {
      newSortDirection = SORT_DIRECTION.ASC;
    } else {
      newSortDirection = SORT_DIRECTION.UNSORTED;
    }

    // eslint-disable-next-line max-len
    const sortedRows = LegendSortedTable._sortRows(this.props.rows, columnType, columnId, newSortDirection);

    this.setState({
      sortedColumnId: columnId,
      sortedColumnType: columnType,
      sortDirection: newSortDirection,
      sortedRows,
    });
  };

  onRowCheckboxClick = (event) => {
    const { checked } = event.target;
    const checkedId = event.target.name;

    const newRowChecks = { ...this.state.rowChecks };
    newRowChecks[checkedId] = checked;

    const { lastCheckedId } = this.state;
    const { sortedRows } = this.state;

    if (event.shiftKey && lastCheckedId) {
      LegendSortedTable._setSelectionRange(newRowChecks, sortedRows, lastCheckedId, checkedId);
    }

    this.setState({
      rowChecks: newRowChecks,
      lastCheckedId: checkedId,
    });
  };

  onAllCheckboxChange = (event) => {
    const allChecked = event.target.checked;

    const newRowChecks = {};

    this.props.rows.forEach((row) => {
      newRowChecks[row.salmonId] = allChecked;
    });

    this.setState({ allChecked, rowChecks: newRowChecks });
  };

  computeChecksArg() {
    const negativeIds = [];
    const positiveIds = [];

    this.props.rows.forEach((row) => {
      const { salmonId } = row;
      let checked = this.state.rowChecks[salmonId];

      if (checked === undefined) {
        checked = row.checked;
      }

      if (checked) {
        positiveIds.push(salmonId);
      } else {
        negativeIds.push(salmonId);
      }
    });

    if (negativeIds.length === 0) {
      return '';
    }

    let sensorsChecksSet;

    if (positiveIds.length < negativeIds.length) {
      sensorsChecksSet = new SensorsChecksSet(true, positiveIds);
    } else {
      sensorsChecksSet = new SensorsChecksSet(false, negativeIds);
    }

    return sensorsChecksSet.makeQueryArgValue();
  }

  renderColumn(type, id, name) {
    const sortDirection = this.state.sortedColumnType === type && this.state.sortedColumnId === `${id}`
      ? this.state.sortDirection
      : SORT_DIRECTION.UNSORTED;

    const className = type === COLUMN_TYPE.NUMBER ? 'legend-number-name' : '';

    return (
      <th
        key={`${type}_${id}`}
        data-column-type={type}
        data-column-id={id}
        data-sort-direction={sortDirection}
        className={className}
        onClick={this.onColumnClick}
      >
        <div className="header-inner">
          {name}
        </div>
      </th>
    );
  }

  renderRow(row) {
    let checked = this.state.rowChecks[row.salmonId];

    if (checked === undefined) {
      checked = row.checked;
    }

    const href = UserLinksBasic.fixOldAdminUrl(row.link);

    return (
      <tr key={row.salmonId} className="solomon-data-row">
        <td>
          <InternalOrExternalLink href={href}>
            <div className="solomon-color-square">
              <div style={{ backgroundColor: row.color }} />
            </div>
          </InternalOrExternalLink>
        </td>
        <td style={{ whiteSpace: 'nowrap' }}>
          <label>
            <input
              type="checkbox"
              name={row.salmonId}
              className="linechecker"
              data-toggle="linechecker"
              checked={checked}
              onClick={this.onRowCheckboxClick}
            />
          </label>
        </td>
        {row.labelColumnValues.map((cell, index) => (
          <td key={index} style={{ whiteSpace: 'nowrap' }}>
            {cell}
          </td>
        ))}
        <td className="legend-number-value legend-value-min">{row.minFormatted}</td>
        <td className="legend-number-value legend-value-max">{row.maxFormatted}</td>
        <td className="legend-number-value legend-value-last">{row.lastFormatted}</td>
        <td className="legend-number-value legend-value-avg">{row.avgFormatted}</td>
        <td className="legend-number-value legend-value-sum">{row.sumFormatted}</td>
      </tr>
    );
  }

  render() {
    return (
      <table className="table table-condensed table-bordered table-striped solomon-legend sorted-table">
        <thead>
          <tr>
            <th className="legend-color-name" />
            <th className="legend-checkbox-column" />
            {this.props.labelColumnNames.map((labelColumnName, index) =>
              this.renderColumn(COLUMN_TYPE.LABEL, index, labelColumnName))}
            {this.renderColumn(COLUMN_TYPE.NUMBER, 'minForSort', 'Min')}
            {this.renderColumn(COLUMN_TYPE.NUMBER, 'maxForSort', 'Max')}
            {this.renderColumn(COLUMN_TYPE.NUMBER, 'lastForSort', 'Last')}
            {this.renderColumn(COLUMN_TYPE.NUMBER, 'avgForSort', 'Avg')}
            {this.renderColumn(COLUMN_TYPE.NUMBER, 'sumForSort', 'Sum')}
          </tr>
        </thead>
        <tbody>
          <tr className="static">
            <td />
            <td>
              <label>
                <input
                  type="checkbox"
                  data-toggle="linechecker-all"
                  data-static-label-id="linechecker-all-top"
                  checked={this.state.allChecked}
                  onChange={this.onAllCheckboxChange}
                />
              </label>
            </td>
            <td>
              {this.props.status}
            </td>
          </tr>
          {this.state.sortedRows.map((row) => this.renderRow(row))}
          <tr className="static">
            <td />
            <td>
              <label>
                <input
                  type="checkbox"
                  data-toggle="linechecker-all"
                  data-static-label-id="linechecker-all-bottom"
                  checked={this.state.allChecked}
                  onChange={this.onAllCheckboxChange}
                />
              </label>
            </td>
            <td>
              {this.props.status}
            </td>
          </tr>
        </tbody>
      </table>
    );
  }
}

LegendSortedTable.propTypes = {
  rows: PropTypes.array.isRequired,
  labelColumnNames: PropTypes.array.isRequired,
  status: PropTypes.string.isRequired,
};

export default LegendSortedTable;
