/* eslint-disable no-restricted-syntax */
import GraphSettings from './GraphSettings';
import UserLinksBasic from '../../../../utils/UserLinksBasic';
import { parseSearch } from '../../../../utils/url';
import { isTrue } from '../../../../utils/boolean';

class GraphConfUtils {
  static getMainGraphMode(graphConf) {
    return GraphConfUtils.getGraphMode(graphConf, true);
  }

  static hasNonAggregatedGraph(graphConf) {
    return !GraphConfUtils.getGraphModeAggregated(graphConf, true)
      || !GraphConfUtils.getGraphModeAggregated(graphConf, false);
  }

  static getGraphMode(graphConf, primary) {
    const graphModeSetting = primary ? GraphSettings.graphMode : GraphSettings.secondaryGraphMode;
    let graphMode = graphConf[graphModeSetting.queryArg];
    if (!graphMode) {
      graphMode = graphModeSetting.getDefaultValue();
    }
    return graphMode;
  }

  static getInterpolate(graphConf) {
    const parameterValue = graphConf[GraphSettings.interpolate.queryArg];

    if (!parameterValue) {
      return 'linear';
    }

    return parameterValue;
  }

  static isHideNoData(graphConf) {
    const hideNoData = graphConf[UserLinksBasic.HIDE_NO_DATA_QUERY_ARG];
    if (!hideNoData) {
      return false;
    }
    return isTrue(hideNoData);
  }

  static hasTransform(graphConf) {
    const transform = graphConf[UserLinksBasic.TRANSFORM];
    return !!transform && GraphSettings.transform.defaultValue !== transform;
  }

  static hasOverLinesTransform(graphConf) {
    const transform = graphConf[UserLinksBasic.OVER_LINES_TRANSFORM];
    return !!transform && GraphSettings.transform.defaultValue !== transform;
  }

  static hasFilter(graphConf) {
    const filter = graphConf[UserLinksBasic.FILTER];
    return !!filter && GraphSettings.filter.defaultValue !== filter;
  }

  static hasOverride(graphConf) {
    return GraphConfUtils.hasFilter(graphConf) || GraphConfUtils.isHideNoData(graphConf);
  }

  static getGraphModeAggregated(graphConf, primary) {
    const mode = GraphConfUtils.getGraphMode(graphConf, primary);
    return mode !== UserLinksBasic.GRAPH_MODE_GRAPH;
  }

  static hasTransformations(graphConf) {
    return GraphConfUtils.hasTransform(graphConf)
      || GraphConfUtils.hasOverLinesTransform(graphConf)
      || GraphConfUtils.hasFilter(graphConf)
      || GraphConfUtils.isHideNoData(graphConf);
  }

  static toChartConf(graphConf) {
    return {
      yaxis: {
        min: graphConf.min,
        max: graphConf.max,
        log: graphConf.scale === UserLinksBasic.SCALE_LOG,
      },
      xaxis: {},
      threshold: graphConf.threshold || null,
    };
  }

  static fromQueryArgs(url) {
    const search = parseSearch(url);

    const r = {};

    for (const setting of GraphSettings.settings) {
      const value = search.get(setting.queryArg) || '';
      if (value) {
        r[setting.queryArg] = value;
      }
    }

    return r;
  }

  static fromQueryArgParams(params) {
    const r = {};

    for (const setting of GraphSettings.settings) {
      const value = params[setting.queryArg];
      if (value) {
        r[setting.queryArg] = value;
      }
    }

    return r;
  }

  static merge(basic, override) {
    const r = {};

    for (const setting of GraphSettings.settings) {
      const settingName = setting.queryArg;

      const a = basic[settingName];
      const b = override[settingName];

      let merged;
      if (b) {
        merged = b;
      } else if (a) {
        merged = a;
      } else {
        merged = '';
      }
      if (merged) {
        r[settingName] = merged;
      }
    }

    return r;
  }
}

export default GraphConfUtils;
