/* eslint-disable max-len,key-spacing */
import * as Booleans from '../../../../utils/boolean';
import GraphSettings from './GraphSettings';

function parseBool(value) {
  return value === null || value === undefined || !value ? null : Booleans.isTrue(value);
}

function parseInt(value) {
  if (value === null || value === undefined || !value) {
    return null;
  }

  try {
    return parseInt(value);
  } catch (e) {
    return null;
  }
}

function parseDouble(value) {
  if (value === null || value === undefined || !value) {
    return null;
  }
  try {
    return parseFloat(value);
  } catch (e) {
    return null;
  }
}

const GraphMode = {
  graph: 'GRAPH',
  pieChart: 'PIE',
  bars: 'BARS',
  distribution: 'DISTRIBUTION',
  heatmap: 'HEATMAP',
  none: 'NONE',
};

const SecondaryGraphMode = {
  pieChart: 'PIE',
  bars: 'BARS',
  distribution: 'DISTRIBUTION',
  heatmap: 'HEATMAP',
  none: 'NONE',
};

const ColorSchemeType = {
  '': 'AUTO',
  default: 'DEFAULT',
  gradient: 'GRADIENT',
};

const AggregationType = {
  avg: 'AVG',
  min: 'MIN',
  max: 'MAX',
  sum: 'SUM',
  last: 'LAST',
};

const DownsamplingAggregationType = {
  // eslint-disable-next-line quote-props
  avg: 'AVG',
  min: 'MIN',
  max: 'MAX',
  sum: 'SUM',
  last: 'LAST',
};

const DownsamplingFillType = {
  // eslint-disable-next-line quote-props
  null: 'NULL',
  none: 'NONE',
  previous: 'PREVIOUS',
};

const InterpolationType = {
  linear: 'LINEAR',
  left: 'LEFT',
  right: 'RIGHT',
  none: 'NONE',
};

const ScaleType = {
  natural: 'NATURAL',
  log: 'LOG',
};

const OverLinesTransform = {
  none: 'NONE',
  PERCENTILE: 'PERCENTILE',
  WEIGHTED_PERCENTILE: 'WEIGHTED_PERCENTILE',
  SUMMARY: 'SUMMARY',
};

const FilterOrder = {
  none: 'NONE',
  top: 'TOP',
  bottom: 'BOTTOM',
};

const FilterFunction = {
  avg: 'AVG',
  min: 'MIN',
  max: 'MAX',
  sum: 'SUM',
  last: 'LAST',
  std: 'STD',
  count: 'COUNT',
  integrate: 'INTEGRATE',
  random: 'RANDOM',
};

const GraphTransform = {
  none: 'NONE',
  differentiate: 'DIFFERENTIATE',
  differentiate_with_negative: 'DIFFERENTIATE_WITH_NEGATIVE',
  integrate: 'INTEGRATE',
  moving_average: 'MOVING_AVERAGE',
  moving_percentile: 'MOVING_PERCENTILE',
  diff: 'DIFF',
  asap: 'ASAP',
};

const DownsamplingMode = {
  auto: 'AUTO',
  byInterval: 'BY_INTERVAL',
  byPoints: 'BY_POINTS',
  off: 'OFF',
};

const graphModeMap = {
  graph : 'graph',
  pie: 'pieChart',
  bars : 'bars',
  distribution: 'distribution',
  heatmap: 'heatmap',
  none: 'none',
};

const secondaryGraphModeMap = {
  pie: 'pieChart',
  bars : 'bars',
  distribution: 'distribution',
  heatmap: 'heatmap',
  none: 'none',
};

function fromValueOrNull(converter, value) {
  if (value in converter) {
    return converter[value];
  }

  return null;
}

class GraphConverter {
  static toModel(proto) {
    const body = {};

    body.description = GraphSettings.description.getFromOrDefault(proto);

    body.graphMode = fromValueOrNull(GraphMode, GraphSettings.graphMode.getFromOrDefault(proto));
    body.secondaryGraphMode = fromValueOrNull(SecondaryGraphMode, GraphSettings.secondaryGraphMode.getFromOrDefault(proto));
    body.min = GraphSettings.min.getFromOrDefault(proto);
    body.max = GraphSettings.max.getFromOrDefault(proto);
    body.normalize = parseBool(GraphSettings.norm.getFromOrDefault(proto));
    body.colorScheme = fromValueOrNull(ColorSchemeType, GraphSettings.cs.getFromOrDefault(proto));
    body.dropNans = parseBool(GraphSettings.dropNans.getFromOrDefault(proto));
    body.stack = parseBool(GraphSettings.stack.getFromOrDefault(proto));
    body.aggr = fromValueOrNull(AggregationType, GraphSettings.aggr.getFromOrDefault(proto));
    body.interpolate = fromValueOrNull(InterpolationType, GraphSettings.interpolate.getFromOrDefault(proto));
    body.scale = fromValueOrNull(ScaleType, GraphSettings.scale.getFromOrDefault(proto));
    body.numberFormat = GraphSettings.numberFormat.getFromOrDefault(proto);
    body.green = GraphSettings.green.getFromOrDefault(proto);
    body.yellow = GraphSettings.yellow.getFromOrDefault(proto);
    body.red = GraphSettings.red.getFromOrDefault(proto);
    body.violet = GraphSettings.violet.getFromOrDefault(proto);
    body.greenValue = GraphSettings.greenValue.getFromOrDefault(proto);
    body.yellowValue = GraphSettings.yellowValue.getFromOrDefault(proto);
    body.redValue = GraphSettings.redValue.getFromOrDefault(proto);
    body.violetValue = GraphSettings.violetValue.getFromOrDefault(proto);
    body.overLinesTransform = fromValueOrNull(OverLinesTransform, GraphSettings.overLinesTransform.getFromOrDefault(proto));
    body.percentiles = GraphSettings.percentiles.getFromOrDefault(proto);
    body.bucketLabel = GraphSettings.bucketLabel.getFromOrDefault(proto);
    body.filter = fromValueOrNull(FilterOrder, GraphSettings.filter.getFromOrDefault(proto));
    body.filterBy = fromValueOrNull(FilterFunction, GraphSettings.filterBy.getFromOrDefault(proto));
    body.filterLimit = GraphSettings.filterLimit.getFromOrDefault(proto);
    body.transform = fromValueOrNull(GraphTransform, GraphSettings.transform.getFromOrDefault(proto));
    body.movingWindow = GraphSettings.movingWindow.getFromOrDefault(proto);
    body.movingPercentile = GraphSettings.movingPercentile.getFromOrDefault(proto);
    body.downsampling = fromValueOrNull(DownsamplingMode, GraphSettings.downsampling.getFromOrDefault(proto));
    body.downsamplingAggr = fromValueOrNull(DownsamplingAggregationType, GraphSettings.downSamplingAggr.getFromOrDefault(proto));
    body.downsamplingFill = fromValueOrNull(DownsamplingFillType, GraphSettings.downsamplingFill.getFromOrDefault(proto));
    body.grid = GraphSettings.grid.getFromOrDefault(proto);
    body.maxPoints = parseInt(GraphSettings.maxPoints.getFromOrDefault(proto));
    body.hideNoData = GraphSettings.hideNoData.getFromOrDefault(proto);
    body.threshold = parseDouble(GraphSettings.threshold.getFromOrDefault(proto));
    return body;
  }

  static fromModel(model) {
    const builder = {};

    if (model.stack !== null && model.stack !== undefined) {
      builder.stack = `${model.stack}`;
    }

    const { dropNans } = model;
    if (dropNans !== undefined) {
      builder.dropNaN = dropNans;
    }

    const { threshold } = model;
    if (threshold !== undefined && threshold !== null) {
      builder.threshold = `${threshold}`;
    }

    builder.description = model.description || '';
    builder.graphMode = model.graphMode
      ? graphModeMap[model.graphMode.toLowerCase()] || ''
      : '';
    builder.secondaryGraphMode = model.secondaryGraphMode
      ? secondaryGraphModeMap[model.secondaryGraphMode.toLowerCase()] || ''
      : '';

    if (model.min) {
      builder.min = model.min;
    }
    if (model.max) {
      builder.max = model.max;
    }
    if (model.normalize !== undefined) {
      builder.norm = `${model.normalize}`;
    }
    if (model.colorScheme && model.colorScheme !== 'auto') {
      builder.cs = model.colorScheme.toLowerCase();
    }
    if (model.aggr) {
      builder.aggr = model.aggr.toLowerCase();
    }
    if (model.interpolate) {
      builder.interpolate = model.interpolate.toLowerCase();
    }
    if (model.overLinesTransform) {
      builder.overLinesTransform = model.overLinesTransform;
    }
    if (model.scale) {
      builder.scale = model.scale.toLowerCase();
    }
    if (model.numberFormat) {
      builder.numberFormat = model.numberFormat;
    }
    if (model.green) {
      builder.green = model.green;
    }
    if (model.yellow) {
      builder.yellow = model.yellow;
    }
    if (model.red) {
      builder.red = model.red;
    }
    if (model.violet) {
      builder.violet = model.violet;
    }
    if (model.greenValue) {
      builder.greenValue = model.greenValue;
    }
    if (model.yellowValue) {
      builder.yellowValue = model.yellowValue;
    }
    if (model.redValue) {
      builder.redValue = model.redValue;
    }
    if (model.violetValue) {
      builder.violetValue = model.violetValue;
    }
    if (model.percentiles) {
      builder.percentiles = model.percentiles;
    }
    if (model.filter) {
      builder.filter = model.filter.toLowerCase();
    }
    if (model.filterBy) {
      builder.filterBy = model.filterBy.toLowerCase();
    }
    if (model.filterLimit) {
      builder.filterLimit = model.filterLimit.toLowerCase();
    }
    if (model.transform) {
      builder.transform = model.transform.toLowerCase();
    }
    if (model.movingWindow) {
      builder.movingWindow = model.movingWindow;
    }
    if (model.movingPercentile) {
      builder.movingPercentile = model.movingPercentile;
    }
    if (model.downsampling) {
      if (model.downsampling === 'AUTO' || model.downsampling === 'DEFAULT') {
        builder.downsampling = 'auto';
      } else if (model.downsampling === 'BY_INTERVAL') {
        builder.downsampling = 'byInterval';
      } else if (model.downsampling === 'BY_POINTS') {
        builder.downsampling = 'byPoints';
      } else if (model.downsampling === 'OFF') {
        builder.downsampling = 'off';
      }
    }
    if (model.downsamplingAggr && model.downsamplingAggr.toLowerCase() !== 'default') {
      builder.downsamplingAggr = model.downsamplingAggr.toLowerCase();
    }
    if (model.downsamplingFill) {
      builder.downsamplingFill = model.downsamplingFill.toLowerCase();
    }
    if (model.ignoreMinStepMillis !== undefined) {
      builder.ignoreMinStepMillis = model.ignoreMinStepMillis;
    }
    if (model.grid) {
      builder.grid = model.grid;
    }
    if (model.maxPoints !== null && model.maxPoints !== undefined && model.maxPoints !== 0) {
      builder.maxPoints = `${model.maxPoints}`;
    }

    if (model.hideNoData !== null && model.hideNoData !== undefined) {
      builder.hideNoData = model.hideNoData;
    }

    return builder;
  }
}

export default GraphConverter;
