/* eslint-disable max-len,no-prototype-builtins,no-param-reassign */
import * as WwwBooleans from '../../../../utils/boolean';
import GraphSettingOption from './GraphSettingOption';
import { DEFAULT_NUMBER_FORMAT_VALUE } from '../../../graphs/NumberFormatUtils';

export const ControlKind = {
  BOOLEAN_DROPDOWN: 0,
  DROPDOWN: 1,
  TEXTFIELD: 2,
  MULTILINETEXT: 3,
  NUMBER_FORMAT: 4,
};

function normalize(value) {
  return value.replace('\r', '');
}

class GraphSetting {
  static DEFAULT_ON_LABEL = 'On';

  static DEFAULT_OFF_LABEL = 'Off';

  kind;

  queryArg;

  title;

  hint;

  options;

  defaultValue;

  showIf;

  constructor(kind, queryArg, title, hint, options, defaultValue, showIf) {
    this.kind = kind;
    this.queryArg = queryArg;
    this.title = title;
    this.hint = hint;
    this.options = options;
    this.defaultValue = defaultValue;
    this.showIf = showIf;
  }

  getDefaultValue() {
    return this.defaultValue;
  }

  getRawOrEmpty(source) {
    const r = source[this.queryArg];
    if (r === undefined || r === null) {
      return '';
    }

    return `${r}`;
  }

  isDefaultValue(value, defaults) {
    if (!value) {
      return true;
    }

    if (!(value instanceof String)) {
      value = `${value}`;
    }

    let defaultValue = this.getRawOrEmpty(defaults);
    if (!defaultValue) {
      defaultValue = this.defaultValue;
    }

    const normalizedValue = normalize(value);
    defaultValue = normalize(defaultValue);

    return normalizedValue === defaultValue;
  }

  static textField(queryArg, title, hint, defaultValue = '', showIf = null) {
    return new GraphSetting(ControlKind.TEXTFIELD, queryArg, title, hint, [], defaultValue, showIf);
  }

  static multiLine(queryArg, title, hint) {
    return new GraphSetting(ControlKind.MULTILINETEXT, queryArg, title, hint, [], '', null);
  }

  static numberFormat(queryArg, title, hint) {
    return new GraphSetting(ControlKind.NUMBER_FORMAT, queryArg, title, hint, [], DEFAULT_NUMBER_FORMAT_VALUE, null);
  }

  static boolDropdown(
    queryArg,
    title,
    defaultValue,
    showIf = null,
    onLabel = GraphSetting.DEFAULT_ON_LABEL,
    offLabel = GraphSetting.DEFAULT_OFF_LABEL,
  ) {
    const options = [
      new GraphSettingOption('', `Default (${defaultValue ? onLabel : offLabel})`),
      new GraphSettingOption('true', onLabel),
      new GraphSettingOption('false', offLabel),
    ];

    return new GraphSetting(ControlKind.BOOLEAN_DROPDOWN, queryArg, title, '', options, '', showIf);
  }

  static dropdown(queryArg, title, showIf, options) {
    return new GraphSetting(ControlKind.DROPDOWN, queryArg, title, '', options, options[0].getValue(), showIf);
  }

  isShow(source) {
    return this.showIf ? this.showIf.call(null, source) : true;
  }

  getFromOrDefault(source) {
    let r = source[this.queryArg];
    if (r === undefined || r === null) {
      r = this.defaultValue;
    }

    return `${r}`;
  }

  getFromOrDefaultStrictly(source) {
    let r = source[this.queryArg];
    if (r === undefined || r === null) {
      r = this.defaultValue;
    }

    if (this.kind === ControlKind.DROPDOWN) {
      for (let i = 0; i < this.options.length; ++i) {
        const optionValue = this.options[i].value;
        if (optionValue.toLowerCase() === r.toLowerCase()) {
          return optionValue;
        }
      }

      return this.defaultValue;
    }

    return `${r}`;
  }

  getBoolFrom(source) {
    return WwwBooleans.isTrue(this.getFromOrDefault(source));
  }
}

export default GraphSetting;
