/* eslint-disable no-restricted-syntax,no-use-before-define */
/* eslint-disable max-len */
import GraphSettingOption from './GraphSettingOption';
import GraphSetting from './GraphSetting';
import UserLinksBasic from '../../../../utils/UserLinksBasic';

function isGradient(source) {
  return GraphSettings.cs.getFromOrDefault(source) === UserLinksBasic.COLOR_SCHEME_GRADIENT;
}

function isAnyGraphOfType(source, type) {
  return GraphSettings.graphMode.getFromOrDefault(source) === type
    || GraphSettings.secondaryGraphMode.getFromOrDefault(source) === type;
}

function isAnyGraphHeatmap(source) {
  return isAnyGraphOfType(source, UserLinksBasic.GRAPH_MODE_HEATMAP);
}

function isAnyGraphLines(source) {
  return isAnyGraphOfType(source, UserLinksBasic.GRAPH_MODE_GRAPH);
}

function isOverLinesPercentile(source) {
  const param = GraphSettings.overLinesTransform.getFromOrDefault(source);
  return param === 'PERCENTILE' || param === 'WEIGHTED_PERCENTILE';
}

const GraphMode = {
  GRAPH: { value: UserLinksBasic.GRAPH_MODE_GRAPH, title: 'Polyline', canBeSecondary: false },
  GRAPH_MODE_PIE_CHART: { value: UserLinksBasic.GRAPH_MODE_PIE_CHART, title: 'Pie chart', canBeSecondary: true },
  GRAPH_MODE_BARS: { value: UserLinksBasic.GRAPH_MODE_BARS, title: 'Bars', canBeSecondary: true },
  GRAPH_MODE_DISTRIBUTION: { value: UserLinksBasic.GRAPH_MODE_DISTRIBUTION, title: 'Distribution', canBeSecondary: true },
  GRAPH_MODE_HEATMAP: { value: UserLinksBasic.GRAPH_MODE_HEATMAP, title: 'Heatmap', canBeSecondary: true },
  GRAPH_MODE_NONE: { value: UserLinksBasic.GRAPH_MODE_NONE, title: 'None', canBeSecondary: true },
};

const GraphModes = [
  GraphMode.GRAPH,
  GraphMode.GRAPH_MODE_PIE_CHART,
  GraphMode.GRAPH_MODE_BARS,
  GraphMode.GRAPH_MODE_DISTRIBUTION,
  GraphMode.GRAPH_MODE_HEATMAP,
  GraphMode.GRAPH_MODE_NONE,
];

const graphModeOptions = GraphModes.map((mode) => new GraphSettingOption(mode.value, mode.title));
const secondaryGraphModeOptions = GraphModes.filter((mode) => mode.canBeSecondary).map((mode) => new GraphSettingOption(mode.value, mode.title));

class GraphSettings {
  static description = GraphSetting.multiLine(UserLinksBasic.GRAPH_DESCRIPTION, 'Graph description', 'description');

  static max = GraphSetting.textField(UserLinksBasic.MAX_PARAM, 'Max', '19G');

  static min = GraphSetting.textField(UserLinksBasic.MIN_PARAM, 'Min', '17K');

  static norm = GraphSetting.boolDropdown(UserLinksBasic.NORM_PARAM, 'Normalize', false);

  static cs = GraphSetting.dropdown(UserLinksBasic.COLOR_SCHEME_QUERY_ARG, 'Gradient', null, [
    new GraphSettingOption('', 'Auto'),
    new GraphSettingOption('default', 'Default'),
    new GraphSettingOption('gradient', 'Gradient'),
  ]);

  static dropNans = GraphSetting.boolDropdown(UserLinksBasic.DROP_NANS_PARAM, 'Drop NaN value', false);

  static stack = GraphSetting.boolDropdown(UserLinksBasic.STACK_PARAM, 'Stack/Area', true, isAnyGraphLines);

  static aggr = GraphSetting.dropdown(UserLinksBasic.AGGR_PARAM, 'Aggregation function', null, [
    new GraphSettingOption('avg', 'avg'),
    new GraphSettingOption('min', 'min'),
    new GraphSettingOption('max', 'max'),
    new GraphSettingOption('sum', 'sum'),
    new GraphSettingOption('last', 'last'),
  ]);

  static graphMode = GraphSetting.dropdown(UserLinksBasic.GRAPH_MODE, 'Main graph', null, graphModeOptions);

  static secondaryGraphMode = GraphSetting.dropdown(UserLinksBasic.SECONDARY_GRAPH_MODE, 'Secondary graph', null, secondaryGraphModeOptions);

  static interpolate = GraphSetting.dropdown(UserLinksBasic.INTERPOLATE_PARAM, 'Interpolate', null, [
    new GraphSettingOption('linear', 'Linear'),
    new GraphSettingOption('left', 'Left'),
    new GraphSettingOption('right', 'Right'),
    new GraphSettingOption('none', 'None'),
  ]);

  static transform = GraphSetting.dropdown(
    UserLinksBasic.TRANSFORM,
    'Transform',
    isAnyGraphLines,
    [
      new GraphSettingOption('none', 'None'),
      new GraphSettingOption('differentiate', 'Differentiate'),
      new GraphSettingOption('differentiate_with_negative', 'Differentiate with negative'),
      new GraphSettingOption('integrate', 'Integrate'),
      new GraphSettingOption('moving_average', 'Moving average'),
      new GraphSettingOption('moving_percentile', 'Moving percentile'),
      new GraphSettingOption('diff', 'Diff between values'),
      new GraphSettingOption('asap', 'Auto Smoother'),
    ],
  );

  static overLinesTransform = GraphSetting.dropdown(UserLinksBasic.OVER_LINES_TRANSFORM,
    'Over lines transform',
    isAnyGraphLines,
    [
      new GraphSettingOption('NONE', 'None'),
      new GraphSettingOption('PERCENTILE', 'Percentile'),
      new GraphSettingOption('WEIGHTED_PERCENTILE', 'Weighted Percentile'),
      new GraphSettingOption('SUMMARY', 'Summary'),

    ]);

  static scale = GraphSetting.dropdown(UserLinksBasic.SCALE_PARAM, 'Scale', null, [
    new GraphSettingOption(UserLinksBasic.SCALE_NATURAL, 'Natural'),
    new GraphSettingOption(UserLinksBasic.SCALE_LOG, 'Logarithmic'),
  ]);

  static numberFormat = GraphSetting.numberFormat(UserLinksBasic.GRAPH_NUMBER_FORMAT, 'Number format', '.0,K.3');

  static green = GraphSetting.textField(UserLinksBasic.GREEN_QUERY_ARG, 'Green', 'label', '', isGradient);

  static yellow = GraphSetting.textField(UserLinksBasic.YELLOW_QUERY_ARG, 'Yellow', 'label', '', isGradient);

  static red = GraphSetting.textField(UserLinksBasic.RED_QUERY_ARG, 'Red', 'label', '', isGradient);

  static violet = GraphSetting.textField(UserLinksBasic.VIOLET_QUERY_ARG, 'Violet', 'label', '', isGradient);

  static greenValue = GraphSetting.textField(UserLinksBasic.HEATMAP_GREEN_VALUE_QUERY_ARG, 'Green', '10k', '', isAnyGraphHeatmap);

  static yellowValue = GraphSetting.textField(UserLinksBasic.HEATMAP_YELLOW_VALUE_QUERY_ARG, 'Yellow', '20m', '', isAnyGraphHeatmap);

  static redValue = GraphSetting.textField(UserLinksBasic.HEATMAP_RED_VALUE_QUERY_ARG, 'Red', '30g', '', isAnyGraphHeatmap);

  static violetValue = GraphSetting.textField(UserLinksBasic.HEATMAP_VIOLET_VALUE_QUERY_ARG, 'Violet', '40g', '', isAnyGraphHeatmap);

  static percentiles = GraphSetting.textField(UserLinksBasic.PERCENTILE_PARAM, 'Percentiles', '50,90,99', '50,90,99', isOverLinesPercentile);

  static bucketLabel = GraphSetting.textField(UserLinksBasic.BUCKET_LABEL_PARAM, 'Bucket label', 'bin', '', (source) => GraphSettings.overLinesTransform.getRawOrEmpty(source) === 'WEIGHTED_PERCENTILE');

  static filter = GraphSetting.dropdown(
    UserLinksBasic.FILTER,
    'Filter',
    null,
    [
      new GraphSettingOption('none', 'None'),
      new GraphSettingOption('top', 'Top'),
      new GraphSettingOption('bottom', 'Bottom'),
    ],
  );

  static filterBy = GraphSetting.dropdown(
    UserLinksBasic.FILTER_BY,
    'Filter by',
    (source) => {
      const filterKind = GraphSettings.filter.getFromOrDefault(source);
      return filterKind === 'top' || filterKind === 'bottom';
    },
    [
      new GraphSettingOption('avg', 'avg'),
      new GraphSettingOption('min', 'min'),
      new GraphSettingOption('max', 'max'),
      new GraphSettingOption('sum', 'sum'),
      new GraphSettingOption('last', 'last'),
      new GraphSettingOption('std', 'std'),
      new GraphSettingOption('count', 'count'),
      new GraphSettingOption('integrate', 'integrate'),
      new GraphSettingOption('random', 'random'),
    ],
  );

  static filterLimit = GraphSetting.textField(
    UserLinksBasic.FILTER_LIMIT,
    'Filter Limit',
    '3',
    '3',
    (source) => {
      const filterKind = GraphSettings.filter.getFromOrDefault(source);
      return filterKind === 'top' || filterKind === 'bottom';
    },
  );

  static downsampling = GraphSetting.dropdown(
    UserLinksBasic.DOWNSAMPLING,
    'Downsampling',
    null,
    [
      new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_AUTO, 'Auto'), // Default
      new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_WITHOUT_GRID, 'Without grid'),
      new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL, 'By interval'),
      new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS, 'By points'),
      new GraphSettingOption(UserLinksBasic.DOWNSAMPLING_PARAM_OFF, 'Off'),
    ],
  );

  static grid = GraphSetting.textField(
    UserLinksBasic.GRID,
    'Grid interval',
    '1m30s',
    '',
    (s) => UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL === GraphSettings.downsampling.getFromOrDefault(s),
  );

  // We must support points and maxPoints parameters both,
  // otherwise 'points' from args overrides 'points' from graph proto config
  static maxPoints = GraphSetting.textField(
    UserLinksBasic.MAX_POINTS,
    'Max points',
    '100',
    '100',
    (s) => UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS === GraphSettings.downsampling.getFromOrDefault(s),
  );

  static downSamplingAggr = GraphSetting.dropdown(
    UserLinksBasic.DOWNSAMPLING_AGGR,
    'Aggregation function use for downsampling',
    (s) => UserLinksBasic.DOWNSAMPLING_PARAM_OFF !== GraphSettings.downsampling.getFromOrDefault(s),
    [
      new GraphSettingOption('default', 'default'),
      new GraphSettingOption('avg', 'avg'),
      new GraphSettingOption('max', 'max'),
      new GraphSettingOption('min', 'min'),
      new GraphSettingOption('sum', 'sum'),
      new GraphSettingOption('last', 'last'),
    ],
  );

  static downsamplingFill = GraphSetting.dropdown(
    UserLinksBasic.DOWNSAMPLING_FILL,
    'Fill option',
    (s) => UserLinksBasic.DOWNSAMPLING_PARAM_OFF !== GraphSettings.downsampling.getFromOrDefault(s),
    [
      new GraphSettingOption('null', 'Null'),
      new GraphSettingOption('none', 'None'),
      new GraphSettingOption('previous', 'Previous'),
    ],
  );

  // Deprecated
  static ignoreMinStepMillis = GraphSetting.boolDropdown(
    UserLinksBasic.IGNORE_MIN_STEP_MILLIS,
    'Ignore min step interval',
    false,
    (s) => UserLinksBasic.DOWNSAMPLING_PARAM_OFF !== GraphSettings.downsampling.getFromOrDefault(s),
  );

  static movingWindow = GraphSetting.textField(
    UserLinksBasic.MOVING_WINDOW,
    'Moving window',
    '5m',
    '5m',
    (s) => GraphSettings.transform.getFromOrDefault(s).startsWith('moving_'),
  );

  static movingPercentile = GraphSetting.textField(
    UserLinksBasic.MOVING_PERCENTILE,
    'Percentile',
    '80',
    '80',
    (s) => GraphSettings.transform.getFromOrDefault(s) === 'moving_percentile',
  );

  static hideNoData = GraphSetting.boolDropdown(UserLinksBasic.HIDE_NO_DATA_QUERY_ARG, 'Hide no data', false);

  static threshold = GraphSetting.textField('threshold', 'Threshold', '100.2');

  static showGraphsLegend = GraphSetting.boolDropdown(UserLinksBasic.LEGEND_QUERY_ARG, 'Show graphs legend', false);

  static settings = [
    GraphSettings.description,
    GraphSettings.graphMode,
    GraphSettings.secondaryGraphMode,
    GraphSettings.stack,
    GraphSettings.scale,
    GraphSettings.interpolate,
    GraphSettings.aggr,
    GraphSettings.dropNans,
    GraphSettings.numberFormat,
    GraphSettings.cs,
    GraphSettings.green,
    GraphSettings.yellow,
    GraphSettings.red,
    GraphSettings.violet,
    GraphSettings.greenValue,
    GraphSettings.yellowValue,
    GraphSettings.redValue,
    GraphSettings.violetValue,
    GraphSettings.norm,
    GraphSettings.min,
    GraphSettings.max,
    GraphSettings.transform,
    GraphSettings.movingWindow,
    GraphSettings.movingPercentile,
    GraphSettings.overLinesTransform,
    GraphSettings.percentiles,
    GraphSettings.bucketLabel,
    GraphSettings.filter,
    GraphSettings.filterBy,
    GraphSettings.filterLimit,
    GraphSettings.downsampling,
    GraphSettings.grid,
    GraphSettings.maxPoints,
    GraphSettings.downSamplingAggr,
    GraphSettings.downsamplingFill,
    GraphSettings.ignoreMinStepMillis,
    GraphSettings.hideNoData,
    GraphSettings.threshold,
  ];

  static dashboardSettings = [
    GraphSettings.showGraphsLegend,
  ];

  static getByName(settings, name) {
    for (const setting of settings) {
      if (setting.queryArg === name) {
        return setting;
      }
    }
    throw new Error(`setting not found by name: ${name}`);
  }
}

export default GraphSettings;
