import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import OldHeader from '../../../components/Header/OldHeader';
import OldPageTitle from '../../../components/OldPageTitle';
import TopLevelAlerts from '../../../components/TopLevelAlerts';
import MonitoringButton from '../components/ControlBar/MonitoringButton';

import PinsBlock from '../components/PinsBlock/PinsBlock';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import ProjectsBlock from '../../projects/ProjectsBlock';
import { getMonitoringUiHost } from '../utils/graphToExpressionTransformations';
import { isPreOrProdIntEnv } from '../../../utils/env';

import { fetchProjects } from '../../../store/reducers/projects/projects';
import { getPins, deletePin } from '../../../store/reducers/old/userPins';

import './style.css';

const mapStateToParams = (state) => {
  const params = {};
  const { text, page, pageSize } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }

  params.pageSize = pageSize;

  return params;
};

class OldProjectsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { text: '', page: 0, pageSize: 50 };
  }

  componentDidMount() {
    this.projectFilter.focus();
    this.props.fetchProjects(mapStateToParams(this.state));
    this.props.getPins({ url: '/' });
  }

  onFilterChange = (event) => {
    this.doReload({ text: event.target.value });
    this.setState({ filter: event.target.value });
  };

  onFilterKeyPress = (event) => {
    const { filter } = this.state;
    if (event.which === 13 && filter) {
      const projects = this.props.projectsPage.result || [];

      if (projects.length === 1) {
        const singleProjectId = projects[0].id;
        const url = UserLinksBasic.fixOldAdminUrl(`/?project=${singleProjectId}`);
        this.context.router.history.push(url);
      }
    }
  };

  onActivePageChange = (activePage) => {
    this.doReload({ page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ pageSize, page: 0 });
  };

  onPinDelete = (id) => {
    this.props.deletePin(id)
      .then(() => this.props.getPins({ url: '/' }));
  };

  doReload = (change) => {
    this.setState(change, () => {
      this.props.fetchProjects(mapStateToParams(this.state));
    });
  };

  render() {
    const { projectsPage, favoriteProjects } = this.props;

    const newUiUrl = `https://${getMonitoringUiHost()}/?utm_source=solomon_main_page`;

    return (
      <div>
        <OldPageTitle title="Home" />
        <OldHeader />
        <div className="container-fluid container-solomon">
          <TopLevelAlerts admin={false} />
          <h3>Home</h3>
          <div className="row">
            <div className="col-xs-12">
              <input
                type="text"
                className="labelsBlock-filter-apply"
                placeholder="Filter projects on page"
                value={this.state.text}
                onKeyPress={this.onFilterKeyPress}
                onChange={this.onFilterChange}
                ref={(input) => {
                  this.projectFilter = input;
                }}
              />
              {isPreOrProdIntEnv() && (
                <span style={{ marginLeft: '2em' }}>
                  <MonitoringButton href={newUiUrl} />
                </span>
              )}
            </div>
            <div className="col-xs-8">
              <div className="labelBlocks">
                <ProjectsBlock
                  projectsPage={projectsPage}
                  favoriteProjects={favoriteProjects}
                  text={this.state.text}
                  isAdmin={false}
                  onActivePageChange={this.onActivePageChange}
                  onPageSizeChange={this.onPageSizeChange}
                />
              </div>
            </div>
            <div className="col-xs-4">
              <PinsBlock pins={this.props.userPins} onDelete={this.onPinDelete} />
            </div>
          </div>
        </div>
      </div>
    );
  }
}

OldProjectsPage.propTypes = {
  projectsPage: PropTypes.object.isRequired,
  favoriteProjects: PropTypes.array.isRequired,
  userPins: PropTypes.array.isRequired,
  fetchProjects: PropTypes.func.isRequired,
  getPins: PropTypes.func.isRequired,
  deletePin: PropTypes.func.isRequired,
};

OldProjectsPage.contextTypes = {
  router: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  projectsPage: state.projects,
  favoriteProjects: state.favoriteProjects.projects,
  userPins: state.userPins,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  fetchProjects,
  getPins,
  deletePin,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(OldProjectsPage);
