import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';

import OldPageTitle from '../../../components/OldPageTitle';
import OldHeader from '../../../components/Header/OldHeader';
import TopLevelAlerts from '../../../components/TopLevelAlerts';
import { formatSearch, parseSearch } from '../../../utils/url';

import LabelGroup from '../../../components/LabelGroups/LabelGroup';
import AutoGraphBlock from './AutoGraphBlock';
import debounce from '../../../utils/debounce';

import getSelectorsFromQueryArgs from '../../../utils/queryArgs';
import Selector, { isGlobValue } from '../../../utils/Selector';
import UserLinksBasic from '../../../utils/UserLinksBasic';

import PinsBlock from '../components/PinsBlock/PinsBlock';
import MatchingItemsBlock from './MatchingItemsBlock';
import WarningAlert from '../../projects/alerts/WarningAlert';

import OldSelectors from '../components/OldSelectors/OldSelectors';
import UrlUtils from '../../../utils/url/UrlUtils';
import Selectors from '../../../utils/Selectors';
import { formatOldSelectorEntriesAsNewSelectors } from '../../../utils/SelectorUtils';

import {
  getSensorsPage, getFilteredLabels, deletePin, clearSensorsPage,
} from '../../../store/reducers/old/oldSensorsPage';
import { mustShowClusterServiceOnly } from '../../metrics/mustShowClusterServiceOnly';
import BrowserUtils from '../../../utils/BrowserUtils';
import YasmAlert from '../../../components/YasmAlert';
import FavoriteProjectToggler from '../../projects/FavoriteProjectToggler';
import MonitoringButton from '../components/ControlBar/MonitoringButton';
import { isPreOrProdIntEnv } from '../../../utils/env';

function isOldGlobValue(value) {
  return value.startsWith('!') || value === '-' || isGlobValue(value);
}

class OldSensorsPage extends PureComponent {
  static mapSearchToState(search) {
    const params = parseSearch(search);

    const project = params.get('project') || '';
    const cluster = params.get('cluster') || '';
    const service = params.get('service') || '';

    params.delete('project');
    params.delete('cluster');
    params.delete('service');

    const selectorsList = getSelectorsFromQueryArgs(params);

    const selectorsList2 = [...selectorsList];
    if (cluster) {
      selectorsList2.push(['cluster', cluster]);
    }
    if (service) {
      selectorsList2.push(['service', service]);
    }
    const selectors = formatOldSelectorEntriesAsNewSelectors(selectorsList2);
    const showClusterServiceOnly = mustShowClusterServiceOnly(selectors);

    const forceCluster = params.get(UserLinksBasic.FORCE_CLUSTER_PARAM)
      || params.get(UserLinksBasic.FORCE_HOST_PARAM)
      || '';

    return {
      project, cluster, service, selectors: selectorsList, showClusterServiceOnly, forceCluster, text: '',
    };
  }

  static mapStateToSearch(state) {
    const search = {};

    search.project = state.project;

    if (state.cluster) {
      search.cluster = state.cluster;
    }

    if (state.service) {
      search.service = state.service;
    }

    state.selectors.forEach((s) => {
      const key = s[0];
      const value = s[1];
      const keyWithPrefix = `l.${key}`;
      search[keyWithPrefix] = value;
    });

    if (state.forceCluster) {
      search.forceCluster = state.forceCluster;
    }

    return `?${formatSearch(search)}`;
  }

  static mapStateToParams(state) {
    const params = {};

    const selectorsList = [];
    if (state.cluster) {
      selectorsList.push(['cluster', state.cluster]);
    }
    if (state.service) {
      selectorsList.push(['service', state.service]);
    }

    if (state.showClusterServiceOnly) {
      if (selectorsList.length > 0) {
        params.selectors = formatOldSelectorEntriesAsNewSelectors(selectorsList);
      }
    } else {
      selectorsList.push(...state.selectors);

      params.selectors = formatOldSelectorEntriesAsNewSelectors(selectorsList);
    }

    if (state.forceCluster) {
      params.forceCluster = state.forceCluster;
    }

    if (state.showClusterServiceOnly) {
      params.names = 'cluster,service';
    }

    if (state.text) {
      params.text = state.text;
    }

    return params;
  }

  static mapStateToMatchingParams(state) {
    const params = {};

    let selectorsList;

    if (state.showClusterServiceOnly) {
      selectorsList = [['project', state.project]];
      if (state.cluster) {
        selectorsList.push(['cluster', state.cluster]);
      }
      if (state.service) {
        selectorsList.push(['service', state.service]);
      }
    } else {
      const exactInsideShardLabels = state.selectors.filter((s) => !isOldGlobValue(s[1]));

      selectorsList = [
        ['project', state.project],
        ['cluster', state.cluster],
        ['service', state.service],
        ...exactInsideShardLabels,
      ];
    }

    if (selectorsList.length > 0) {
      const parsedSelectorsList = selectorsList.map((s) => Selector.glob(s[0], s[1]));
      params.labels = new Selectors('', parsedSelectorsList).format();
    }

    return params;
  }

  constructor(props) {
    super(props);
    this.state = OldSensorsPage.mapSearchToState(props.location.search);
  }

  componentDidMount() {
    this.loadData();
  }

  componentDidUpdate(prevProps) {
    if (this._filterEl) {
      this._filterEl.focus();
    }
    if (this.props.location.search !== prevProps.location.search) {
      const newState = OldSensorsPage.mapSearchToState(this.props.location.search);
      this.setState(newState, () => this.loadData());
    }
  }

  componentWillUnmount() {
    this.props.clearSensorsPage();
  }

  formatWithNewLabelValue = (label, value) => {
    if (label === 'cluster' || label === 'service') {
      const newState = { ...this.state, [label]: value };
      return OldSensorsPage.mapStateToSearch(newState);
    }

    let hasLabels = false;

    const selectors = this.state.selectors.map((s) => {
      if (s[0] === label) {
        hasLabels = true;
        return [label, value];
      }

      return s;
    });

    if (!hasLabels) {
      selectors.push([label, value]);
    }

    const newState = { ...this.state, selectors };
    return OldSensorsPage.mapStateToSearch(newState);
  };

  handleTextChange = (event) => {
    event.preventDefault();
    this.setState({ text: event.target.value }, () => this.loadFilteredLabelsDebounced());
  };

  handleDeletePin = (id) => {
    this.props.deletePin(id);
  };

  doReload = () => {
    this.props.history.push(OldSensorsPage.mapStateToSearch(this.state));
  };

  loadData = () => {
    const projectId = this.state.project;
    const params = OldSensorsPage.mapStateToParams(this.state);
    const matchingParams = OldSensorsPage.mapStateToMatchingParams(this.state);
    this.props.getSensorsPage(projectId, params, matchingParams, this.props.location.search);
  };

  loadFilteredLabels = () => {
    const projectId = this.state.project;
    const params = OldSensorsPage.mapStateToParams(this.state);
    this.props.getFilteredLabels(projectId, params);
  };

  // eslint-disable-next-line react/sort-comp
  loadFilteredLabelsDebounced = debounce(this.loadFilteredLabels, 200);

  _selectors() {
    // eslint-disable-next-line max-len
    const selectorsAsEntries = getSelectorsFromQueryArgs(parseSearch(this.props.location.search), true, true);

    const pageUrl = selectorsAsEntries
      .map((s) => `${s[0]}=${UrlUtils.encodeArg(s[1])}`)
      .join('&');

    const selectors = selectorsAsEntries
      .filter((s) => s[0] !== 'project')
      .map((selector) => {
        const name = selector[0];
        const value = selector[1];

        const dropPath = UrlUtils.removeQueryArgs(`/?${pageUrl}`, name);

        const requestPath = selectorsAsEntries
          .filter((s) => s[0] !== name)
          .map((s) => `${s[0]}=${encodeURIComponent(s[1])}`)
          .join('&');

        const wildcardType = name === 'cluster' || name === 'service' ? 'NONE' : 'FULL_AND_SIMPLE';

        return ({
          name,
          value,
          asyncSelector: true,
          options: [],
          dropPath,
          requestPath,
          wildcardType,
        });
      });

    return <OldSelectors selectors={selectors} />;
  }

  render() {
    const {
      project, cluster, service, showClusterServiceOnly,
    } = this.state;
    const { search } = this.props.location;

    let title;
    let subTitle = null;

    if (showClusterServiceOnly) {
      if (!cluster && !service) {
        title = project;
        subTitle = <FavoriteProjectToggler projectId={project} />;
      } else if (!cluster && service) {
        title = service;
      } else {
        title = cluster;
      }
    } else {
      title = 'Select labels:';
    }

    const { loading, data, error } = this.props.oldSensorsPage;

    let mainElement = null;
    let menu = {};

    if (data) {
      menu = data.matchingMenus;

      const labelsPage = data.labels;

      const sensorsCount = labelsPage.maxCount;

      const monitoringButton = isPreOrProdIntEnv() && (
        <MonitoringButton
          href={`${window.location.href}&graph=auto&__force_new=1&utm_source=solomon_view_metrics`}
        />
      );

      let nonCriticalAlert = null;
      let singleSensorGraphBlock = null;
      let searchBlock = null;
      let autoGraphBlock = null;
      let labelBlocks = null;

      const selectors = this._selectors();

      if (!showClusterServiceOnly && sensorsCount === 0 && labelsPage.labels.length === 0) {
        nonCriticalAlert = (
          <div className="alert alert-warning">
            No metrics found, try removing some labels
          </div>
        );
      } else if (!showClusterServiceOnly && sensorsCount === 1) {
        title = (
          <>
            Single metric
            {' '}
            <span style={{ marginLeft: '0.5em' }}>{monitoringButton}</span>
          </>
        );

        const autoGraphLink = UserLinksBasic.fixOldAdminUrl(`${search}&graph=auto`);

        singleSensorGraphBlock = (
          <div style={{ marginBottom: '1em' }}>
            Single metric matches:
            {' '}
            <Link to={autoGraphLink}>graph</Link>
          </div>
        );

        labelBlocks = null;
      } else {
        const { labels } = labelsPage;

        const nonTrivialLabels = labels.filter((l) => l.values.length + (l.absent ? 1 : 0) === 1);

        const isSingleNonTrivialLabel = nonTrivialLabels.length === 1;

        autoGraphBlock = showClusterServiceOnly ? null : (
          <AutoGraphBlock
            search={search}
            isSingleNonTrivialLabel={isSingleNonTrivialLabel}
            sensorsCount={sensorsCount}
          />
        );

        searchBlock = (
          <>
            <div className="btn-group" style={{ marginRight: '2em' }}>
              <input
                type="text"
                name="text"
                value={this.state.text}
                placeholder="Filter labels on page..."
                onChange={this.handleTextChange}
                autoComplete="off"
                ref={(el) => {
                  this._filterEl = el;
                }}
              />
            </div>
            {monitoringButton}
          </>
        );

        labelBlocks = labels.map((labelGroup) => (
          <LabelGroup
            key={labelGroup.name}
            name={labelGroup.name}
            values={labelGroup.values}
            absent={labelGroup.absent}
            truncated={labelGroup.truncated}
            canGroupByLabel={labelGroup.name !== 'cluster' && labelGroup.name !== 'service'}
            formatUrl={this.formatWithNewLabelValue}
          />
        ));
      }

      const sidebarBlock = (
        <div className="col-xs-4">
          <MatchingItemsBlock
            projectId={project}
            title="Dashboards"
            type="dashboard"
            items={data.matchingDashboards}
          />
          <MatchingItemsBlock
            projectId={project}
            title="Graphs"
            type="graph"
            items={data.matchingGraphs}
          />
          <PinsBlock pins={data.userPins} onDelete={this.handleDeletePin} />
        </div>
      );

      let yasmAlert = null;
      if (BrowserUtils.getLogicalQueryArgOrEmpty('project').startsWith('yasm_')) {
        yasmAlert = <YasmAlert />;
      }

      mainElement = (
        <div>
          <TopLevelAlerts admin={false} />
          {yasmAlert}
          {selectors}
          <h3>
            {title}
            {subTitle != null && (
              <>
                &nbsp;
                <small>{subTitle}</small>
              </>
            )}
          </h3>
          {nonCriticalAlert}
          {singleSensorGraphBlock}
          <div className="row col-xs-12">{searchBlock}</div>
          <div className="row solomon-labels-dashboards-graphs">
            <div className="col-xs-8">
              {autoGraphBlock}
              {labelBlocks}
            </div>
            {sidebarBlock}
          </div>
        </div>
      );
    } else if (error) {
      mainElement = <WarningAlert title="Error" message={error} />;
    } else if (loading) {
      mainElement = <div>Loading...</div>;
    }

    return (
      <div>
        <OldPageTitle title={title} />
        <OldHeader menu={menu} />
        <div className="container-fluid container-solomon generic-block">
          {mainElement}
        </div>
      </div>
    );
  }
}

OldSensorsPage.propTypes = {
  history: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  oldSensorsPage: PropTypes.object.isRequired,
  getSensorsPage: PropTypes.func.isRequired,
  getFilteredLabels: PropTypes.func.isRequired,
  deletePin: PropTypes.func.isRequired,
  clearSensorsPage: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  oldSensorsPage: state.oldSensorsPage,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  getSensorsPage,
  getFilteredLabels,
  deletePin,
  clearSensorsPage,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(OldSensorsPage);
