import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import Modal from '../../../../components/Modal';
import getSelectorsFromQueryArgs from '../../../../utils/queryArgs';
import { parseSearch } from '../../../../utils/url';
import { formatOldSelectorEntriesAsNewSelectors } from '../../../../utils/SelectorUtils';
import FormInput from '../../../../components/forms/FormInput';
import FormRadioGroup from '../../../graphs/FormRadioGroup';
import {
  ALERT_TYPE_OPTIONS,
  ALERT_TYPES,
  COMPARISON_OPTIONS,
  DEFAULT_THRESHOLD_TYPE,
  THRESHOLD_TYPE_OPTIONS,
} from '../../../alerts/constants';
import FormElement from '../../../../components/forms/FormElement';
import FormArea from '../../../../components/forms/FormArea';
import BrowserUtils from '../../../../utils/BrowserUtils';

function generateSelectors() {
  const selectorEntries = getSelectorsFromQueryArgs(
    parseSearch(window.location.search),
    true,
    true,
  );

  return formatOldSelectorEntriesAsNewSelectors(selectorEntries, true);
}

function generateProgram(newSelectors) {
  return `let source = ${newSelectors};\n// You can set own check expression here\nalarm_if(last(series_sum(source)) > 0);`;
}

class AlertCreationModal extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      projectId: this.props.defaultProjectId,
      type: ALERT_TYPES.THRESHOLD,
      thresholdTimeAggregation: DEFAULT_THRESHOLD_TYPE,
      thresholdPredicate: 'GT',
      thresholdNumber: 0,
    };
  }

  onApply = () => {
    if (this.state.projectId) {
      const ownerProjectId = this.state.projectId;

      const newSelectors = generateSelectors();

      const type = {};

      if (this.state.type === ALERT_TYPES.THRESHOLD) {
        type.threshold = {
          selectors: newSelectors,
          timeAggregation: this.state.thresholdTimeAggregation,
          predicate: this.state.thresholdPredicate,
          threshold: this.state.thresholdNumber,
        };
      } else if (this.state.type === ALERT_TYPES.EXPRESSION) {
        type.expression = {
          program: generateProgram(newSelectors),
        };
      }

      const graphLink = BrowserUtils.locationHref();

      const body = {
        id: '',
        name: '',
        projectId: ownerProjectId,
        type,
        annotations: { graphLink },
        _isFromAutoGraph: true,
      };

      const newUrl = `/admin/projects/${ownerProjectId}/alerts/new`;

      this.props.onApply(newUrl, body);
    }
  };

  onInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }
    this.setState({ [event.target.name]: event.target.value });
  };

  render() {
    const selectors = generateSelectors();

    let typeBlock;

    switch (this.state.type) {
      case ALERT_TYPES.THRESHOLD: {
        typeBlock = (
          <div>
            <FormElement label="">
              Compare values of a metrics with a user defined threshold
            </FormElement>
            <FormArea
              name="selectors" label="Selectors"
              rows={3}
              value={selectors}
              onChange={this.onInputChange}
              help="Label selectors to define metrics to check"
              disabled
            />
            <FormRadioGroup
              name="thresholdTimeAggregation"
              label="Aggregation"
              options={THRESHOLD_TYPE_OPTIONS}
              value={this.state.thresholdTimeAggregation}
              defaultValue={DEFAULT_THRESHOLD_TYPE}
              onChange={this.onInputChange}
              help="Define criteria by that will be check alert"
            />
            <FormRadioGroup
              name="thresholdPredicate"
              label="Comparison"
              options={COMPARISON_OPTIONS}
              value={this.state.thresholdPredicate}
              defaultValue="GT"
              onChange={this.onInputChange}
              help="Predicate uses for compare point with threshold"
            />
            <FormInput
              type="float"
              name="thresholdNumber"
              label="Threshold"
              value={this.state.thresholdNumber}
              onChange={this.onInputChange}
              help="Target threshold that will be use with metrics data on specified period"
            />
          </div>
        );
        break;
      }
      case ALERT_TYPES.EXPRESSION: {
        const program = generateProgram(selectors);

        typeBlock = (
          <div>
            <FormElement label="">
              Evaluate used defined expression for metric
            </FormElement>
            <FormArea
              name="program"
              label="Program"
              value={program}
              rows={6}
              onChange={this.onInputChange}
              help="Contains expression to evaluate"
              disabled
            />
          </div>
        );
        break;
      }
      default:
        typeBlock = (
          <FormElement label="">
            Please select one of alert types!
          </FormElement>
        );
    }

    return (
      <Modal
        title="Create new alert"
        okTitle="Create"
        isOpen={this.props.isOpen}
        onOk={this.onApply}
        onCancel={this.props.onCancel}
      >
        <div className="form-horizontal">
          <FormInput
            type="text"
            label="Project"
            name="projectId"
            value={this.state.projectId}
            onChange={this.onInputChange}
            help="Select project where you want to save new alert"
          />
          <FormRadioGroup
            name="type"
            label="Type"
            value={this.state.type}
            defaultValue={ALERT_TYPES.THRESHOLD}
            options={ALERT_TYPE_OPTIONS}
            onChange={this.onInputChange}
          />
          {typeBlock}
        </div>
      </Modal>
    );
  }
}

AlertCreationModal.propTypes = {
  defaultProjectId: PropTypes.string.isRequired,
  isOpen: PropTypes.bool.isRequired,
  onApply: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

export default AlertCreationModal;
