import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import Modal from '../../../../components/Modal';
import BrowserUtils from '../../../../utils/BrowserUtils';
import GraphConfUtils from '../../OldGraphPage/utils/GraphConfUtils';
import GraphSettings from '../../OldGraphPage/utils/GraphSettings';
import UserLinksBasic from '../../../../utils/UserLinksBasic';
import getSelectorsFromQueryArgs from '../../../../utils/queryArgs';
import { parseSearch } from '../../../../utils/url';
import { formatOldSelectorEntriesAsNewSelectors } from '../../../../utils/SelectorUtils';
import FormInput from '../../../../components/forms/FormInput';
import GraphConverter from '../../OldGraphPage/utils/GraphConverter';

class ConfiguredGraphCreationModal extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { projectId: this.props.defaultProjectId };
  }

  onApply = () => {
    if (this.state.projectId) {
      const ownerProjectId = this.state.projectId;

      const source = GraphConfUtils.fromQueryArgs(BrowserUtils.getLogicalUrlAfterHost());

      const selectorEntries = getSelectorsFromQueryArgs(
        parseSearch(window.location.search),
        false,
        true,
      );

      const newSelectors = formatOldSelectorEntriesAsNewSelectors(selectorEntries, true);

      const predefinedBody = GraphConverter.toModel(source);

      const body = {
        ...predefinedBody,
        id: '',
        name: '',
        projectId: ownerProjectId,
        description: GraphSettings.description.getFromOrDefault(source),
        parameters: [
          { name: 'project', value: BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.PROJECT) },
          { name: 'cluster', value: BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.CLUSTER) },
          { name: 'service', value: BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.SERVICE) },
        ],
        elements: [
          {
            type: 'EXPRESSION',
            expression: newSelectors,
          },
        ],
        _isFromAutoGraph: true,
      };

      const newUrl = `/admin/projects/${ownerProjectId}/graphs/new`;

      this.props.onApply(newUrl, body);
    }
  };

  onProjectChange = (event) => {
    this.setState({ projectId: event.target.value });
  };

  render() {
    return (
      <Modal
        title="Create new configured graph"
        okTitle="Create"
        isOpen={this.props.isOpen}
        onOk={this.onApply}
        onCancel={this.props.onCancel}
      >
        <div className="form-horizontal">
          <FormInput
            type="text"
            label="Project"
            name="projectId"
            value={this.state.projectId}
            onChange={this.onProjectChange}
            help="Select project where you want to save new configured graph"
          />
        </div>
      </Modal>
    );
  }
}

ConfiguredGraphCreationModal.propTypes = {
  defaultProjectId: PropTypes.string.isRequired,
  isOpen: PropTypes.bool.isRequired,
  onApply: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

export default ConfiguredGraphCreationModal;
