/* eslint-disable jsx-a11y/label-has-for,jsx-a11y/label-has-associated-control */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import Modal from '../../../../../components/Modal';
import BrowserUtils from '../../../../../utils/BrowserUtils';
import UserLinksBasic from '../../../../../utils/UserLinksBasic';
import UrlUtils from '../../../../../utils/url/UrlUtils';
import CopyLinkButton from './CopyLinkButton';
import ShortLinkFormGroup from './ShortLinkFormGroup';
import { formatOldSelectorEntriesAsNewSelectors } from '../../../../../utils/SelectorUtils';
import getSelectorsFromQueryArgs from '../../../../../utils/queryArgs';
import { parseSearch } from '../../../../../utils/url';
import GraphConfUtils from '../../../OldGraphPage/utils/GraphConfUtils';

function isAutoGraph(fullItem) {
  const graphParam = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM);

  if (UserLinksBasic.GRAPH_VALUE_AUTO !== graphParam) {
    return false;
  }

  if (!fullItem) {
    return false;
  }

  if (GraphConfUtils.hasTransformations(fullItem.graphConf)) {
    return false;
  }

  return true;
}

function getStaticOrDynamicUrl(timeRange, linkStatic, baseHashUrl) {
  let staticOrDynamicUrl;

  if (linkStatic) {
    staticOrDynamicUrl = UrlUtils.addTwoParameters(
      baseHashUrl,
      UserLinksBasic.B_QA, timeRange.begin.instant,
      UserLinksBasic.E_QA, timeRange.end.instant,
    );
  } else {
    const b = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.B_QA);
    const e = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.E_QA);

    staticOrDynamicUrl = UrlUtils.addTwoParameters(
      baseHashUrl,
      UserLinksBasic.B_QA, b,
      UserLinksBasic.E_QA, e,
    );
  }

  return staticOrDynamicUrl;
}

function getApiHostname() {
  if (BrowserUtils.isLocalHost() || window.location.hostname.startsWith('solomon-pre')) {
    return 'solomon-pre.yandex.net';
  } if (window.location.hostname.startsWith('solomon-test.')) {
    return 'solomon-test.yandex.net';
  }
  return 'solomon.yandex.net';
}

function makeDataApiCurl(timeRange) {
  try {
    const graphParam = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM);

    if (graphParam !== UserLinksBasic.GRAPH_VALUE_AUTO) {
      return '';
    }

    const projectId = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.PROJECT);
    const methodUrl = `http://${getApiHostname()}/api/v2/projects/${projectId}/sensors/data?maxPoints=500&from=${encodeURIComponent(timeRange.begin.instant)}&to=${encodeURIComponent(timeRange.end.instant)}`;

    const selectorEntries = getSelectorsFromQueryArgs(
      parseSearch(window.location.search),
      false,
      true,
    );

    const newSelectors = formatOldSelectorEntriesAsNewSelectors(selectorEntries, true);

    return `$ curl -X POST \\
    -H 'Content-Type: text/plain' \\
    -H 'Accept: application/json' \\
    -H 'Authorization: OAuth <YOUR_TOKEN>' \\
    -d '${newSelectors}' \\
    '${methodUrl}'`;
  } catch (e) {
    console.error('failed to make Data API curl', e);
    return '';
  }
}

function computeFields(fullItem, timeRange, linkStatic) {
  // populate constants
  const baseHashUrl = UrlUtils.removeQueryArgs(BrowserUtils.getLogicalUrlAfterHost(),
    UserLinksBasic.B_QA, UserLinksBasic.E_QA,
    UserLinksBasic.GRAPH_ONLY_QA,
    UserLinksBasic.AUTOREFRESH_QA);

  const plainHashUrl = getStaticOrDynamicUrl(timeRange, linkStatic, baseHashUrl);

  const embedHashUrl = UrlUtils.addParameter(plainHashUrl,
    UserLinksBasic.GRAPH_ONLY_QA, 'y');

  const commonPart = `${window.location.protocol}//${window.location.host}`;
  const plainUrl = commonPart + plainHashUrl;
  const embedUrl = commonPart + embedHashUrl;
  const iframe = `<iframe height='400' width='100%' src='${embedUrl}'></iframe>`;
  const wiki = `{{iframe height='400' width='100%' src='${embedUrl}' frameborder='0'}}`;

  const curlCommand = makeDataApiCurl(timeRange);

  return {
    plainUrl, iframe, wiki, curlCommand,
  };
}

class OldShareModal extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { linkStatic: true };
  }

  onLinkStaticChange = (event) => {
    this.setState({ linkStatic: event.target.checked });
  };

  render() {
    const { fullItem, timeRange } = this.props;

    const fields = computeFields(fullItem, timeRange, this.state.linkStatic);

    const showExportToChartsButton = isAutoGraph(fullItem);

    const chartsUrl = `https://charts.yandex-team.ru/preview/editor/solomon/autograph/autograph${window.location.search}`;

    return (
      <Modal
        title="Share"
        onCancel={this.props.onCancel}
        isOpen
      >
        <div className="form-horizontal">
          <div className="form-group">
            <label className="col-sm-3 control-label">Static?</label>
            <div className="col-sm-7">
              <div className="checkbox">
                <label>
                  <input
                    className="link-static-checkbox"
                    type="checkbox"
                    checked={this.state.linkStatic}
                    onChange={this.onLinkStaticChange}
                  />
                </label>
              </div>
            </div>
          </div>
          <div className="form-group link-plain">
            <label className="col-sm-3 control-label">Link</label>
            <div className="col-sm-7">
              <input className="form-control" type="text" readOnly="readonly" value={fields.plainUrl} />
            </div>
            <div className="col-sm-2 control-label" style={{ textAlign: 'left' }}>
              <CopyLinkButton value={fields.plainUrl} />
            </div>
          </div>
          <ShortLinkFormGroup value={fields.plainUrl} />
          <div className="form-group link-embed">
            <label className="col-sm-3 control-label">Iframe code</label>
            <div className="col-sm-7">
              <input className="form-control" type="text" readOnly="readonly" value={fields.iframe} />
            </div>
            <div className="col-sm-2 control-label" style={{ textAlign: 'left' }}>
              <CopyLinkButton value={fields.iframe} />
            </div>
          </div>
          <div className="form-group link-wiki">
            <label className="col-sm-3 control-label">Wiki code</label>
            <div className="col-sm-7">
              <input className="form-control" type="text" readOnly="readonly" value={fields.wiki} />
            </div>
            <div className="col-sm-2 control-label" style={{ textAlign: 'left' }}>
              <CopyLinkButton value={fields.wiki} />
            </div>
          </div>
          {showExportToChartsButton && (
            <div className="form-group">
              <label className="col-sm-3 control-label" />
              <div className="col-sm-7">
                <a
                  className="btn btn-default"
                  href={chartsUrl}
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  Export to Charts
                </a>
              </div>
            </div>
          )}
          {fields.curlCommand ? (
            <div className="form-group link-data-api">
              <label className="col-sm-3 control-label">Data API commands</label>
              <div className="col-sm-7">
                <textarea
                  className="form-control"
                  rows={4}
                  readOnly="readonly"
                  style={{ resize: 'none', overflow: 'auto', whiteSpace: 'pre' }}
                  value={fields.curlCommand}
                />
              </div>
            </div>
          ) : null}
        </div>
      </Modal>
    );
  }
}

OldShareModal.propTypes = {
  fullItem: PropTypes.object,
  timeRange: PropTypes.object.isRequired,
  onCancel: PropTypes.func.isRequired,
};

OldShareModal.defaultProps = {
  fullItem: null,
};

export default OldShareModal;
