import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import Datalist from '../../../../components/Datalist';
import getSelectorsFromQueryArgs from '../../../../utils/queryArgs';
import { parseSearch } from '../../../../utils/url';
import { formatOldSelectorEntriesAsNewSelectors } from '../../../../utils/SelectorUtils';
import BrowserUtils from '../../../../utils/BrowserUtils';
import OldSelector from './OldSelector';
import UrlUtils from '../../../../utils/url/UrlUtils';

function getAutoGraphSelectors(search) {
  const selectorEntries = getSelectorsFromQueryArgs(parseSearch(search), true, true);
  return formatOldSelectorEntriesAsNewSelectors(selectorEntries, true);
}

function createSelector(label, search) {
  const selectorEntries = getSelectorsFromQueryArgs(parseSearch(search), true, true);

  const selectorsAsEntriesWithProject = [...selectorEntries];

  const pageUrlWithoutEntityParam = selectorsAsEntriesWithProject
    .map((s) => `${s[0]}=${UrlUtils.encodeArg(s[1])}`)
    .join('&');

  const dropPath = UrlUtils.removeQueryArgs(`/?${pageUrlWithoutEntityParam}`, label);

  const requestPath = selectorsAsEntriesWithProject
    .filter((s) => s[0] !== label)
    .map((s) => `${s[0]}=${encodeURIComponent(s[1])}`)
    .join('&');

  return {
    name: label,
    dropPath,
    asyncSelector: true,
    // Not async parameters
    options: [],
    // Async parameters
    value: '',
    requestPath,
    wildcardType: label === 'cluster' || label === 'service' ? 'NONE' : 'FULL_AND_SIMPLE',
  };
}

class LastPlusSelector extends PureComponent {
  componentDidUpdate(prevProps) {
    if (this.props.search !== prevProps.search) {
      this.props.onHideLabelsDropdown();
    }
  }

  handleShowLabelsDropdown = () => {
    const projectId = BrowserUtils.getLogicalQueryArgOrEmpty('project');
    const newSelectors = getAutoGraphSelectors(this.props.search);
    this.props.onShowLabelsDropdown(projectId, newSelectors);
  };

  handleFilterChange = (event) => {
    const projectId = BrowserUtils.getLogicalQueryArgOrEmpty('project');
    const newSelectors = getAutoGraphSelectors(this.props.search);
    const filter = event.target.value;
    this.props.onChangeLabelFilter(projectId, newSelectors, filter);
  };

  handleSelectLabel = (labelKey) => {
    const selector = createSelector(labelKey, this.props.search);
    this.props.onSelectLabel(selector);
  };

  handleHideLabelsDropdown = () => {
    this.props.onHideLabelsDropdown();
  };

  handleHideLastSelector = () => {
    this.props.onHideLabelsDropdown();
  };

  render() {
    const { lastPlusSelector } = this.props;
    const {
      type, selector, labelFilter, labelOptions,
    } = lastPlusSelector;

    if (type === 'key') {
      // render as datalist
      return (
        <div className="selector">
          <div className="label-selectpicker-placeholder">
            <Datalist
              size="xs"
              name="new_key"
              value={labelFilter}
              datalist={labelOptions}
              placeholder="Select new key"
              onChange={this.handleFilterChange}
              onApply={this.handleSelectLabel}
              onBlur={this.handleHideLabelsDropdown}
              defaultOpened
            />
          </div>
        </div>
      );
    } if (type === 'value' && selector !== null) {
      return (
        <OldSelector
          selector={selector}
          selectorValues={this.props.loadedSelectorsValues[selector.name]}
          showAllValues={this.props.showAllValuesInSelectors}
          getSelectorValues={this.props.getSelectorValues}
          defaultOpened
          onDropPathClick={this.handleHideLastSelector}
        />
      );
    }

    // render as button
    return (
      <div className="selector">
        <div className="label-selectpicker-placeholder">
          <div className="btn-group label-selectpicker">
            <button
              type="button"
              className="btn btn-default btn-xs"
              onClick={this.handleShowLabelsDropdown}
            >
              <span className="filter-option pull-left">
                +
              </span>
            </button>
          </div>
        </div>
      </div>
    );
  }
}

LastPlusSelector.propTypes = {
  search: PropTypes.string.isRequired,
  lastPlusSelector: PropTypes.object.isRequired,
  onShowLabelsDropdown: PropTypes.func.isRequired,
  onHideLabelsDropdown: PropTypes.func.isRequired,
  onChangeLabelFilter: PropTypes.func.isRequired,
  onSelectLabel: PropTypes.func.isRequired,
  loadedSelectorsValues: PropTypes.object.isRequired,
  showAllValuesInSelectors: PropTypes.bool.isRequired,
  getSelectorValues: PropTypes.func.isRequired,
};

export default LastPlusSelector;
