import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import LabelSelector from '../../../../components/LabelSelector/LabelSelector';
import BrowserUtils from '../../../../utils/BrowserUtils';
import UrlUtils from '../../../../utils/url/UrlUtils';
import UserLinksBasic from '../../../../utils/UserLinksBasic';

const SelectorOptionProps = PropTypes.shape({
  value: PropTypes.string,
  fullPath: PropTypes.string.isRequired,
  selected: PropTypes.bool.isRequired,
});

const SelectorProps = PropTypes.shape({
  name: PropTypes.string,
  dropPath: PropTypes.string,
  asyncSelector: PropTypes.bool,
  // Not async parameters
  options: PropTypes.arrayOf(SelectorOptionProps).isRequired,
  // Async parameters
  value: PropTypes.string,
  requestPath: PropTypes.string,
  wildcardType: PropTypes.string,
});

function prepareSelectorValuesParams(selector, filter, showAllValues) {
  const { name } = selector;

  let selectors;
  let requestPath;

  if (showAllValues) {
    const maps = UrlUtils.parseQueryArgs(selector.requestPath);

    const project = maps.get('project') || '';
    const cluster = maps.get('cluster') || '';
    const service = maps.get('service') || '';

    if (name === 'cluster' || name === 'service' || !cluster || !service) {
      requestPath = `project=${project}`;
    } else {
      requestPath = `project=${project}&cluster=${cluster}&service=${service}`;
    }
  } else {
    requestPath = selector.requestPath;
  }

  if (!filter) {
    selectors = requestPath;
  } else {
    const query = `${filter.startsWith('!') ? '' : '*'}${filter.replace(/\\/g, /\\\\/g)}*`;
    selectors = `${requestPath}&${name}=${encodeURIComponent(query)}`;
  }

  const params = {
    label: name,
    selectors,
    wildcardType: selector.wildcardType,
  };

  const forceCluster = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_CLUSTER_PARAM)
    || BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_HOST_PARAM);

  if (!forceCluster || forceCluster !== 'localhost') {
    params.forceCluster = forceCluster;
  }

  return params;
}

function updateUrlArg(name, value) {
  const url = BrowserUtils.getLogicalUrlAfterHost();
  return UrlUtils.updateOrRemoveParameterWithPrefix(url, name, value);
}

class OldSelector extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { filter: '' };
  }

  onSelect = (value) => {
    if (this.props.selector.asyncSelector && value) {
      const newUrl = updateUrlArg(this.props.selector.name, value);
      this.context.router.history.push(newUrl);
    }
  };

  onLoad = ({ filter, showAllLabels }) => {
    if (this.props.selector.asyncSelector) {
      const params = prepareSelectorValuesParams(this.props.selector, filter, showAllLabels);
      this.props.getSelectorValues(params, BrowserUtils.getLogicalUrlAfterHost());
    } else {
      this.setState({ filter });
    }
  };

  render() {
    const { selector } = this.props;

    let selectedValue = '';
    let loading = false;
    let options;
    let errorMessage = '';

    if (selector.asyncSelector) {
      const selectorValues = (this.props.selectorValues || {});
      loading = selectorValues.loading || false;
      options = selectorValues.data || [];
      errorMessage = selectorValues.error || '';
      selectedValue = selector.value;
    } else {
      options = selector.options.map((option) => ({
        ...option, fullPath: UserLinksBasic.fixOldAdminUrl(option.fullPath),
      }));

      const selectedOptions = options.filter((option) => option.selected);

      if (selectedOptions.length > 0) {
        selectedValue = selectedOptions[0].value;
      }

      if (this.state.filter) {
        const lowerFilter = this.state.filter.toLowerCase();
        options = options.filter((option) => option.value.toLowerCase().includes(lowerFilter));
      }
    }

    const dropPath = UserLinksBasic.fixOldAdminUrl(selector.dropPath);

    return (
      <LabelSelector
        isAsync={selector.asyncSelector}
        defaultOpened={this.props.defaultOpened}
        showAllValues={this.props.showAllValues}
        name={selector.name}
        value={selectedValue}
        dropPath={dropPath}
        loading={loading}
        options={options}
        errorMessage={errorMessage}
        onSelect={this.onSelect}
        onLoad={this.onLoad}
        onDropPathClick={this.props.onDropPathClick}
      />
    );
  }
}

OldSelector.propTypes = {
  selector: SelectorProps.isRequired,
  defaultOpened: PropTypes.bool,
  selectorValues: PropTypes.object,
  showAllValues: PropTypes.bool.isRequired,
  getSelectorValues: PropTypes.func.isRequired,
  onDropPathClick: PropTypes.func,
};

OldSelector.contextTypes = {
  router: PropTypes.object,
};

OldSelector.defaultProps = {
  selectorValues: {},
  defaultOpened: false,
  onDropPathClick: () => {},
};

export default OldSelector;
