/* eslint-disable no-param-reassign,guard-for-in,no-restricted-syntax,max-len */
import GraphSettings from '../OldGraphPage/utils/GraphSettings';
import UserLinksBasic from '../../../utils/UserLinksBasic';
import BrowserUtils from '../../../utils/BrowserUtils';
import LinkedHashMap from '../../../utils/LinkedHashMap';
import UrlUtils from '../../../utils/url/UrlUtils';
import PROJECT_CONFIGS from './projectConfigs';
import { doubleQuote } from '../../../utils/Quoter';
import { convertOldToNewSelectors, formatOldSelectorEntriesAsNewSelectors } from '../../../utils/SelectorUtils';
import StringInterpolator from '../../../utils/StringInterpolator';
import { LOAD1_PART_TYPE, replaceSelectorsInCode, SELECTORS_PART_TYPE } from '../../../utils/SelectorsReplacer';
import Selector from '../../../utils/Selector';
import Selectors from '../../../utils/Selectors';
import parseKmg from '../../../utils/Kmg';
import { getSelectorsAsEntriesFromParams } from '../../../utils/queryArgs';
import { isPreEnv, isProdEnv } from '../../../utils/env';
import { computeRangeFromParamsForOldUi } from '../../data/utils';
import { parseDuration } from '../../../utils/duration';

function computeOverLinesTransform(source) {
  const overLinesTransform = GraphSettings.overLinesTransform.getFromOrDefault(source);
  const percentiles = GraphSettings.percentiles.getRawOrEmpty(source);

  // Necessary to support ...&graph=auto&percentiles=50,90,99
  // without overLinesTransofrm=WEIGHTED_PERCENTILE
  if (overLinesTransform === 'NONE' && !!percentiles) {
    return 'WEIGHTED_PERCENTILE';
  }

  return overLinesTransform;
}

export function computeStackForAutoGraph(project, source) {
  const overLinesTransform = computeOverLinesTransform(source);
  const hasOverLinesTransform = GraphSettings.overLinesTransform.getDefaultValue()
    .toLowerCase() !== overLinesTransform.toLowerCase();

  if (hasOverLinesTransform) {
    return 'false';
  }

  const stackParam = GraphSettings.stack.getFromOrDefault(source);

  if (stackParam) {
    return stackParam;
  }

  const projectConfig = PROJECT_CONFIGS[project];

  if (!projectConfig) {
    return 'true';
  }

  const projectStack = projectConfig.stack;

  if (projectStack === undefined) {
    return 'true';
  }

  return projectStack;
}

export function computeInterpolationForAutoGraph(project, source) {
  const rawInterpolateParam = GraphSettings.interpolate.getRawOrEmpty(source);

  if (!rawInterpolateParam) {
    const projectConfig = PROJECT_CONFIGS[project];
    if (!projectConfig) {
      return '';
    }
    const projectInterpolate = projectConfig.interpolate;
    return projectInterpolate || '';
  }

  return rawInterpolateParam;
}

function putToParamsIfNeeded(params, paramName) {
  const paramValue = BrowserUtils.getLogicalQueryArgOrEmpty(paramName);

  if (paramValue) {
    params.put(paramName, paramValue);
  }
}

function createSimpleProgram(flattenExprs) {
  return {
    variables: [],
    returnExpr: flattenExprs,
  };
}

export function createProgramWithPercentiles(program, percentiles) {
  if (!percentiles) {
    return createSimpleProgram(program);
  }

  let param;

  if (percentiles.length === 1) {
    param = percentiles[0];
  } else {
    param = `as_vector(${percentiles.join(', ')})`;
  }

  const expr = `to_vector(percentile_group_lines(${param}, ${program}))`;
  return createSimpleProgram(expr);
}

export function createProgramWithHistogramPercentiles(program, percentiles, bucketLabel) {
  if (!percentiles) {
    return createSimpleProgram(program);
  }

  const bucketLabelParam = `${bucketLabel ? `${doubleQuote(bucketLabel)}, ` : ''}`;

  let param;

  if (percentiles.length === 1) {
    param = percentiles[0];
  } else {
    param = `as_vector(${percentiles.join(', ')})`;
  }

  const expr = `histogram_percentile(${param}, ${bucketLabelParam}${program})`;
  return createSimpleProgram(expr);
}

function createProgramWithSummary(program) {
  const summaryAggregates = ['max', 'min', 'sum', 'avg', 'count'];

  const vector = summaryAggregates
    .map((aggr) => `  alias(to_vector(group_lines("${aggr}", ${program})), "${aggr}")`)
    .join(',\n');

  return {
    variables: [],
    returnExpr: `flatten\n(${vector})`,
  };
}

function formatProgramWithVariables(programWithVariables) {
  if (programWithVariables.variables.length === 0) {
    return programWithVariables.returnExpr;
  }

  const variablesBlock = programWithVariables.variables
    .map((v) => `let ${v[0]} = ${v[1]};\n`)
    .join('');
  const returnBlock = `return ${programWithVariables.returnExpr};`;
  return `${variablesBlock}${returnBlock}`;
}

export function transformProgram(source, program) {
  const filter = GraphSettings.filter.getFromOrDefault(source);
  const filterBy = GraphSettings.filterBy.getFromOrDefault(source) || 'avg';
  const filterLimit = GraphSettings.filterLimit.getFromOrDefault(source) || '3';

  if (filter === 'top' || filter === 'bottom') {
    program = `${filter}(${filterLimit}, "${filterBy}", ${program})`;
  }

  const dropNans = GraphSettings.dropNans.getBoolFrom(source);
  if (dropNans) {
    program = `drop_nan(${program})`;
  }

  const transform = GraphSettings.transform.getFromOrDefault(source);

  switch (transform) {
    case 'differentiate':
      program = `non_negative_derivative(${program})`;
      break;
    case 'differentiate_with_negative':
      program = `derivative(${program})`;
      break;
    case 'integrate':
      program = `integral(${program})`;
      break;
    case 'moving_average': {
      const window = GraphSettings.movingWindow.getFromOrDefault(source) || GraphSettings.movingWindow.defaultValue;
      program = `moving_avg(${program}, ${window})`;
      break;
    }
    case 'moving_percentile': {
      const window = GraphSettings.movingWindow.getFromOrDefault(source) || GraphSettings.movingWindow.defaultValue;
      const percentile = GraphSettings.movingPercentile.getFromOrDefault(source) || GraphSettings.movingPercentile.defaultValue;
      program = `moving_percentile(${program}, ${window}, ${percentile})`;
      break;
    }
    case 'diff':
      program = `diff(${program})`;
      break;
    case 'asap':
      program = `asap(${program})`;
      break;
    default:
      break;
  }

  const overLinesTransform = computeOverLinesTransform(source);

  const percentiles = (GraphSettings.percentiles.getFromOrDefault(source) || GraphSettings.percentiles.defaultValue)
    .split(',')
    .map((p) => p.trim());

  const bucketLabel = GraphSettings.bucketLabel.getFromOrDefault(source);

  let programWithVariables;

  switch (overLinesTransform) {
    case 'PERCENTILE': {
      programWithVariables = createProgramWithPercentiles(program, percentiles);
      break;
    }
    case 'WEIGHTED_PERCENTILE':
      programWithVariables = createProgramWithHistogramPercentiles(program, percentiles, bucketLabel);
      break;
    case 'SUMMARY':
      programWithVariables = createProgramWithSummary(program);
      break;
    default:
      programWithVariables = {
        variables: [],
        returnExpr: program,
      };
  }

  const hideNoData = GraphSettings.hideNoData.getBoolFrom(source);

  if (hideNoData) {
    programWithVariables.returnExpr = `drop_empty_lines(${programWithVariables.returnExpr})`;
  }

  return formatProgramWithVariables(programWithVariables);
}

export function createAdminAutoGraphUrl() {
  try {
    const graph = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM);

    if (UserLinksBasic.GRAPH_VALUE_AUTO !== graph) {
      return '';
    }

    const source = BrowserUtils.getLogicalQueryArgsMap().toObject();

    const project = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.PROJECT);

    const selectorsAsEntries = getSelectorsAsEntriesFromParams(source);

    const formattedSelectors = formatOldSelectorEntriesAsNewSelectors(selectorsAsEntries, true);

    const program = transformProgram(source, formattedSelectors);

    const params = new LinkedHashMap();

    if (program === formattedSelectors) {
      params.put('selectors', formattedSelectors);
    } else {
      params.put('expression', program);
    }

    putToParamsIfNeeded(params, UserLinksBasic.B_QA);
    putToParamsIfNeeded(params, UserLinksBasic.E_QA);

    params.put(UserLinksBasic.STACK_PARAM, computeStackForAutoGraph(project, source));

    const interpolate = computeInterpolationForAutoGraph(project, source);
    if (interpolate) {
      params.put(UserLinksBasic.INTERPOLATE_PARAM, interpolate);
    }

    putToParamsIfNeeded(params, UserLinksBasic.SCALE_PARAM);
    putToParamsIfNeeded(params, UserLinksBasic.NORM_PARAM);
    putToParamsIfNeeded(params, UserLinksBasic.COLOR_SCHEME_QUERY_ARG);
    putToParamsIfNeeded(params, UserLinksBasic.GREEN_QUERY_ARG);
    putToParamsIfNeeded(params, UserLinksBasic.YELLOW_QUERY_ARG);
    putToParamsIfNeeded(params, UserLinksBasic.RED_QUERY_ARG);
    putToParamsIfNeeded(params, UserLinksBasic.VIOLET_QUERY_ARG);

    putToParamsIfNeeded(params, UserLinksBasic.MIN_PARAM);
    putToParamsIfNeeded(params, UserLinksBasic.MAX_PARAM);

    const downsamplingMode = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.DOWNSAMPLING);
    if (downsamplingMode) {
      params.put('downsamplingMode', downsamplingMode);
    }

    putToParamsIfNeeded(params, UserLinksBasic.DOWNSAMPLING_AGGR);
    putToParamsIfNeeded(params, UserLinksBasic.DOWNSAMPLING_FILL);
    putToParamsIfNeeded(params, UserLinksBasic.IGNORE_MIN_STEP_MILLIS);
    putToParamsIfNeeded(params, UserLinksBasic.GRID);
    putToParamsIfNeeded(params, UserLinksBasic.MAX_POINTS);

    const adminPath = `/admin/projects/${project}/autoGraph?`;
    return adminPath + UrlUtils.makeQueryArgs(params);
  } catch (e) {
    console.error('failed to create admin autograph url', e);
    return '';
  }
}

export function getMonitoringUiHost() {
  if (isPreEnv()) {
    return 'monitoring-preprod.yandex-team.ru';
  }
  if (isProdEnv()) {
    return 'monitoring.yandex-team.ru';
  }
  return '';
}

function putDownsamplingParamsForNewUi(params, source) {
  const downsamplingType = GraphSettings.downsampling.getFromOrDefaultStrictly(source);
  const downsamplingAggr = GraphSettings.downSamplingAggr.getFromOrDefaultStrictly(source);
  const downsamplingFill = GraphSettings.downsamplingFill.getFromOrDefaultStrictly(source);

  if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_OFF) {
    params.put('dsp_method', 'disabled');
    return;
  }

  const hasAggr = downsamplingAggr && downsamplingAggr !== 'default';
  const hasFill = downsamplingFill && downsamplingFill !== 'null';

  if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS) {
    params.put('dsp_method', 'points');
    params.put('dsp_points', parseInt(source.maxPoints, 10) || 1000);
  } else if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL) {
    params.put('dsp_method', 'interval');
    params.put('dsp_interval', parseDuration(source.grid) || 3600000);
  } else if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_AUTO && (hasAggr || hasFill)) {
    params.put('dsp_method', 'auto');
  }

  if (hasAggr) {
    params.put('dsp_aggr', downsamplingAggr.toLowerCase());
  }
  if (hasFill) {
    params.put('dsp_fill', downsamplingFill.toLowerCase());
  }
}

function putTimeParamsForNewUi(params) {
  const b = BrowserUtils.getLogicalQueryArgOrEmpty('b') || '1d';
  const e = BrowserUtils.getLogicalQueryArgOrEmpty('e');
  const nowMillis = Date.now();

  const { fromMillis, toMillis } = computeRangeFromParamsForOldUi(b, e, nowMillis);

  params.put('from', `${fromMillis}`);
  params.put('to', `${toMillis}`);

  let shortcut = '';
  if (e === '') {
    if (b === '1h' || b === '1d' || b === '1w') {
      shortcut = b;
    }
    if (b === '7d') {
      shortcut = '1w';
    }
    if (b === '31d') {
      shortcut = '1M';
    }
  }
  if (shortcut) {
    params.put('shortcut', shortcut);
  }
}

function putPassthroughParamsForNewUi(params) {
  const utmSource = BrowserUtils.getLogicalQueryArgOrEmpty('utm_source');

  if (utmSource) {
    params.put('utm_source', utmSource);
  }
}

export function createConfiguredGraphOrDashboardUrl(dashboardId) {
  try {
    const host = getMonitoringUiHost();

    if (!host) {
      return '';
    }

    const source = BrowserUtils.getLogicalQueryArgsMap().toObject();

    const project = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.PROJECT);

    const selectorsAsEntries = getSelectorsAsEntriesFromParams(source, false);

    const params = new LinkedHashMap();

    for (const entry of selectorsAsEntries) {
      params.put(`p.${entry[0]}`, entry[1]);
    }

    putTimeParamsForNewUi(params);
    putPassthroughParamsForNewUi(params);

    const path = `https://${host}/projects/${project}/dashboards/${dashboardId}?`;
    return path + UrlUtils.makeQueryArgs(params);
  } catch (e) {
    console.error('failed to create configured graph or dashboard url', dashboardId, e);
    return '';
  }
}

export function createExplorerUrl(isText) {
  const host = getMonitoringUiHost();

  if (!host) {
    return '';
  }

  try {
    const graph = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.GRAPH_PARAM);

    if (UserLinksBasic.GRAPH_VALUE_AUTO !== graph) {
      return '';
    }

    const source = BrowserUtils.getLogicalQueryArgsMap().toObject();

    const project = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.PROJECT);

    const selectorsAsEntries = getSelectorsAsEntriesFromParams(source, true);

    const formattedSelectors = formatOldSelectorEntriesAsNewSelectors(selectorsAsEntries, true);

    const program = transformProgram(source, formattedSelectors);

    const params = new LinkedHashMap();
    const key = isText ? 'q.0.text' : 'q.0.s';
    params.put(key, program);

    const stack = computeStackForAutoGraph(project, source);

    const interpolate = computeInterpolationForAutoGraph(project, source);

    if (source.aggr) {
      params.put('vis_aggr', source.aggr);
    }

    const graphType = source.graphMode;

    if (graphType === 'pieChart') {
      params.put('type', 'pie');
    } else if (stack === 'true') {
      if (interpolate === 'none') {
        params.put('type', 'column');
      } else {
        params.put('type', 'area');
      }
    } else {
      params.put('type', 'line');
    }

    const norm = GraphSettings.interpolate.getBoolFrom(source);
    if (norm) {
      params.put('normz', 'on');
    }

    putDownsamplingParamsForNewUi(params, source);

    putTimeParamsForNewUi(params);
    putPassthroughParamsForNewUi(params);

    const path = `https://${host}/projects/${project}/explorer/queries?`;
    return path + UrlUtils.makeQueryArgs(params);
  } catch (e) {
    console.error('failed to create explorer url', e);
    return '';
  }
}

function isNumberOrKmg(expression) {
  try {
    const num = parseKmg(expression);
    return !isNaN(num);
  } catch (e) {
    return false;
  }
}

function elementToElementWithExpression(element, parametersMap, dataProject, replaceFunc, graphConf) {
  let expression;

  if (element.expression) {
    expression = element.expression;
    // Neccessary to support "40" or "40K" expressions for backward compatibility
    if (isNumberOrKmg(expression)) {
      expression = `constant_line(${expression})`;
    }
  } else if (element.selectors) {
    const oldSelectors = element.selectors.map((s) => `${s.name}=${s.value}`).join('&');
    expression = convertOldToNewSelectors(oldSelectors).format(true);
  } else {
    return null;
  }

  expression = replaceSelectorsInCode(expression, replaceFunc, { projectId: dataProject }).result.join('');

  const dropNaNs = GraphSettings.dropNans.getBoolFrom(graphConf);
  if (dropNaNs) {
    expression = `drop_nan(${expression})`;
  }

  let transform;
  if (element.transform && element.transform !== 'NONE') {
    transform = element.transform;
  } else {
    transform = GraphSettings.transform.getFromOrDefault(graphConf);
  }
  transform = transform.toLowerCase();

  if (transform) {
    switch (transform) {
      case 'none':
        break;
      case 'differentiate':
        expression = `non_negative_derivative(${expression})`;
        break;
      case 'differentiate_with_negative':
        expression = `derivative(${expression})`;
        break;
      case 'integrate':
        expression = `integral(${expression})`;
        break;
      case 'moving_average': {
        const movingWindow = GraphSettings.movingWindow.getFromOrDefault(graphConf) || GraphSettings.movingWindow.defaultValue;
        expression = `moving_avg(${expression}, ${movingWindow})`;
        break;
      }
      case 'moving_percentile': {
        const movingWindow = GraphSettings.movingWindow.getFromOrDefault(graphConf) || GraphSettings.movingWindow.defaultValue;
        const movingPerc = GraphSettings.movingPercentile.getFromOrDefault(graphConf) || GraphSettings.movingPercentile.defaultValue;
        expression = `moving_percentile(${expression}, ${movingWindow}, ${movingPerc})`;
        break;
      }
      case 'diff':
        expression = `diff(${expression})`;
        break;
      case 'asap':
        expression = `asap(${expression})`;
        break;
      default:
      // Do not anything
    }
  }

  if (element && element.title) {
    expression = `alias(${expression}, ${doubleQuote(element.title)})`;
  }

  return { expression, element };
}

function mergeSelectorsAndParams(selectors, params) {
  let result = selectors;

  for (const key in params) {
    if (!result.hasKey(key)) {
      const value = params[key] || '*';
      let selector;
      if (value === '*') {
        selector = Selector.any(key);
      } else if (value === '-') {
        selector = Selector.absent(key);
      } else if (value.startsWith('!')) {
        selector = Selector.notGlob(key, value.substr(1));
      } else {
        selector = Selector.glob(key, value);
      }
      result = result.addOverride(selector);
    }
  }

  return result;
}

const overrideSelectorsByParameters = (params) => (type, text, selectors) => {
  let parsedSelectors;
  if (type === SELECTORS_PART_TYPE) {
    parsedSelectors = Selectors.parse(selectors);
    const interpolatedLabelSelectors = parsedSelectors.getSelectors().map((s) => {
      const interpolatedValue = StringInterpolator.interpolatePattern(s.getValue(), params, false);
      return new Selector(s.getOp(), s.getKey(), interpolatedValue);
    });
    parsedSelectors = new Selectors(parsedSelectors.getNameSelector(), interpolatedLabelSelectors);
  } else {
    const interpolatedOldSelectors = StringInterpolator.interpolatePattern(selectors, params, false);
    parsedSelectors = convertOldToNewSelectors(interpolatedOldSelectors);
  }

  const fixedSelectors = mergeSelectorsAndParams(parsedSelectors, params);

  let fixedSelectorsStr = fixedSelectors.format(true);
  if (type === LOAD1_PART_TYPE) {
    fixedSelectorsStr = `single(${fixedSelectorsStr})`;
  }

  return {
    part: fixedSelectorsStr,
    isOk: true,
  };
};

export function createElementsWithExpressionList(dataProject, parametersMap, graph, graphConf) {
  const replaceFunc = overrideSelectorsByParameters(parametersMap);

  let elementsWithExpressionList = graph.elements
    .map((element) => elementToElementWithExpression(element, parametersMap, dataProject, replaceFunc, graphConf))
    .filter(Boolean);

  const overLinesTransform = GraphSettings.overLinesTransform.getFromOrDefault(graphConf).toLowerCase();
  const filter = GraphSettings.filter.getFromOrDefault(graphConf).toLowerCase();
  const hasOverLinesTransform = overLinesTransform && overLinesTransform !== 'none';
  const hasFilter = filter && filter !== 'none';

  const hideNoData = GraphSettings.hideNoData.getBoolFrom(graphConf);

  if (hasOverLinesTransform || hasFilter) {
    let flattenExprs;

    if (elementsWithExpressionList.length === 1) {
      // Simplify program for single element:
      // - {...} - for single selectors
      // - to_vector({...}) or to_vector(single({...})) - for expression
      const e = elementsWithExpressionList[0];
      if (e.element && e.element.type === 'SELECTORS') {
        flattenExprs = e.expression;
      } else {
        flattenExprs = `to_vector(${e.expression})`;
      }
    } else {
      // Complex flatten expressions:
      // flatten(to_vector({...}), to_vector({...}), to_vector(single({...}))
      flattenExprs = `flatten(${elementsWithExpressionList.map((e) => `to_vector(${e.expression})`).join(', ')})`;
    }

    const filterBy = GraphSettings.filterBy.getFromOrDefault(graphConf).toLowerCase() || 'avg';
    const filterLimit = GraphSettings.filterLimit.getFromOrDefault(graphConf) || '3';

    if (filter === 'top' || filter === 'bottom') {
      flattenExprs = `${filter}(${filterLimit}, "${filterBy}", ${flattenExprs})`;
    }

    const percentiles = (GraphSettings.percentiles.getFromOrDefault(graphConf) || GraphSettings.percentiles.defaultValue)
      .split(',')
      .map((p) => p.trim());

    const bucketLabel = GraphSettings.bucketLabel.getFromOrDefault(graphConf);

    let programWithVariables;

    switch (overLinesTransform) {
      case 'percentile': {
        programWithVariables = createProgramWithPercentiles(flattenExprs, percentiles);
        break;
      }
      case 'weighted_percentile':
        programWithVariables = createProgramWithHistogramPercentiles(flattenExprs, percentiles, bucketLabel);
        break;
      case 'summary': {
        programWithVariables = createProgramWithSummary(flattenExprs);
        break;
      }
      default:
        programWithVariables = createSimpleProgram(flattenExprs);
    }

    if (hideNoData) {
      programWithVariables.returnExpr = `drop_empty_lines(${programWithVariables.returnExpr})`;
    }

    const newExpression = formatProgramWithVariables(programWithVariables);
    const newElement = { type: 'EXPRESSION', expression: newExpression };
    if (hasOverLinesTransform) {
      newElement.area = false;
    }
    elementsWithExpressionList = [{ expression: newExpression, element: newElement }];
  } else if (hideNoData) {
    elementsWithExpressionList = elementsWithExpressionList.map((e) => ({
      expression: `drop_empty_lines(${e.expression})`,
      element: e.element,
    }));
  }

  return elementsWithExpressionList;
}
