/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import TableEditor from '../../components/TableEditor';
import TableViewer from '../../components/TableViewer';
import UserName from '../../components/UserName';

const ACL_ENTRY_PERMISSIONS = {
  READ: { value: 'READ', title: 'Read' },
  WRITE: { value: 'WRITE', title: 'Write' },
  CONFIG_UPDATE: { value: 'CONFIG_UPDATE', title: 'Update' },
  CONFIG_DELETE: { value: 'CONFIG_DELETE', title: 'Delete' },
};

const PERMISSION_TITLES = Object.keys(ACL_ENTRY_PERMISSIONS)
  .map((key) => ACL_ENTRY_PERMISSIONS[key].title);

const PERMISSION_VALUES = Object.keys(ACL_ENTRY_PERMISSIONS)
  .map((key) => ACL_ENTRY_PERMISSIONS[key].value);

const ACL_ENTRY_SHAPE = PropTypes.shape({
  login: PropTypes.string,
  permissions: PropTypes.arrayOf(PropTypes.string),
});

const checkbox = (checked) => (
  checked && (<span className="glyphicon glyphicon-ok" />)
);

function wrapHandler(index, handler) {
  if (!handler) {
    return null;
  }
  return (event) => {
    event.preventDefault();
    event.stopPropagation();
    handler(index);
  };
}

class ReadonlyRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    const { login, permissions } = value;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td><UserName login={login} /></td>
        {PERMISSION_VALUES.map((permission) => (
          <td key={`PERMISSION_${permission}`}>
            {checkbox(permissions.indexOf(permission) >= 0)}
          </td>
        ))}
        {this.onDelete
        && (
        <td className="actions">
          <button type="button" onClick={this.onDelete}>
            <i className="glyphicon glyphicon-remove" />
          </button>
        </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: ACL_ENTRY_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { ...props.value };
  }

  componentDidMount() {
    this.loginInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  onCheckboxChange = (event) => {
    const { name, checked } = event.target;
    const newPermissions = this.state.permissions.slice();
    const index = newPermissions.indexOf(name);
    if (checked && index < 0) {
      newPermissions.push(name);
    } else if (!checked && index >= 0) {
      newPermissions.splice(index, 1);
    }
    this.setState({ permissions: newPermissions });
  };

  doComplete() {
    const { onOk, index } = this.props;
    const { login, permissions } = this.state;
    onOk(index, { login, permissions });
  }

  render() {
    const { index } = this.props;
    const { login, permissions } = this.state;
    return (
      <tr onKeyDown={this.onKeyDown}>
        <td>{index >= 0 ? index + 1 : ''}</td>
        <td>
          <input
            type="text" name="login" className="form-control"
            value={login}
            onChange={this.onInputChange}
            ref={(input) => { this.loginInput = input; }}
          />
        </td>
        {PERMISSION_VALUES.map((permission) => (
          <td key={`PERMISSION_${permission}`}>
            <input
              type="checkbox"
              name={permission}
              key={`PERMISSION_${permission}`}
              checked={permissions.indexOf(permission) >= 0}
              onChange={this.onCheckboxChange}
            />
          </td>
        ))}
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: ACL_ENTRY_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: { login: '', permissions: [] },
};

export const ReadonlyTable = ({ acl, limit }) => (
  <TableViewer
    columns={['#', 'User', ...PERMISSION_TITLES]}
    values={acl}
    row={ReadonlyRow}
    limit={limit}
  />
);

ReadonlyTable.propTypes = {
  acl: PropTypes.arrayOf(ACL_ENTRY_SHAPE),
  limit: PropTypes.number.isRequired,
};

ReadonlyTable.defaultProps = {
  acl: [],
};

export const EditableTable = ({ acl, onDelete, onUpdate }) => (
  <TableEditor
    columns={['#', 'User', ...PERMISSION_TITLES, '']}
    values={acl || []}
    readonlyRow={ReadonlyRow}
    editableRow={EditableRow}
    onDelete={onDelete}
    onUpdate={onUpdate}
  />
);

EditableTable.propTypes = {
  acl: PropTypes.arrayOf(ACL_ENTRY_SHAPE),
  onDelete: PropTypes.func.isRequired,
  onUpdate: PropTypes.func.isRequired,
};

EditableTable.defaultProps = {
  acl: [],
};
/* eslint-enable */
