import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import PermissionAlert from './alerts/PermissionAlert';
import WarningAlert from './alerts/WarningAlert';

import { loadProject } from '../../store/reducers/projects/projectData';
import OldPageTitle from '../../components/OldPageTitle';
import OldHeader from '../../components/Header/OldHeader';

class OldSubProjectPageWrapper extends PureComponent {
  componentDidMount() {
    if (this.props.graphOnly) {
      return;
    }

    const { projectId } = this.props;

    const { project } = this.props.projectData;
    const prevProjectId = isEmpty(project) ? '' : project.id;

    if (projectId !== prevProjectId) {
      this.props.loadProject(projectId);
    }
  }

  render() {
    if (this.props.graphOnly) {
      return this.props.children;
    }

    const {
      loading, project, projectAuth, error,
    } = this.props.projectData;

    let mainElement = null;

    if (loading) {
      mainElement = <div>Loading...</div>;
    } else if (error) {
      if (error.code === 403) {
        mainElement = (
          <PermissionAlert message={error.message} owner={error.owner} />
        );
      } else if (error.message) {
        mainElement = (
          <WarningAlert title="Failed to load project!" message={error.message} />
        );
      } else {
        mainElement = <WarningAlert title="Warning!" message="Failed to load project" />;
      }
    } else if (!projectAuth || !project) {
      mainElement = null;
    } else {
      return this.props.children;
    }

    return (
      <div>
        <OldPageTitle title="Loading..." />
        <OldHeader />
        <div className="container-fluid container-solomon">
          {mainElement}
        </div>
      </div>
    );
  }
}

OldSubProjectPageWrapper.propTypes = {
  projectId: PropTypes.string.isRequired,
  children: PropTypes.element.isRequired,
  graphOnly: PropTypes.bool.isRequired,
  projectData: PropTypes.object.isRequired,
  loadProject: PropTypes.func.isRequired,
};

export default connect(
  (state) => ({ projectData: state.projectData }),
  (dispatch) => bindActionCreators({
    loadProject,
  }, dispatch),
)(OldSubProjectPageWrapper);
