import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ProjectForm from './ProjectForm';
import { parseSearch } from '../../utils/url';
import { wrapSubProjectEditPage } from './wrapSubProjectPage';

import { getProjectIdmRoles } from '../../store/reducers/projects/projectIdmRoles';
import { getProjectFeatureFlags } from '../../store/reducers/projects/projectFeatureFlags';

import { loadProject, saveProject, clearProject } from '../../store/reducers/projects/projectData';

class ProjectEditPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._isNew = !this._projectId;

    if (this._isNew) {
      this._projectId = parseSearch(props.location.search).get('duplicateOf');
    }
  }

  componentDidMount() {
    const { project } = this.props;
    if (this._projectId) {
      const loadedId = project.id;
      if (this._projectId !== loadedId) {
        this.props.loadProject(this._projectId);
      }
      this.props.getProjectFeatureFlags(project.id);
      this.props.getProjectIdmRoles(project.id);
    } else {
      this.props.clearProject();
    }
  }

  onSubmit = (project) => {
    const { history } = this.props;
    this.props.saveProject(project, this._isNew)
      .then(() => history.push(`/admin/projects/${project.id}`));
  };

  render() {
    const project = this.props.project || {};
    const title = this._isNew ? 'New project' : `Project ${project.name}`;
    const isLoading = (this._projectId && isEmpty(project));

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={title} />
        {isLoading && <span>Loading...</span>}
        {!isLoading
          && (
          <ProjectForm
            project={project}
            isNew={this._isNew}
            onSubmit={this.onSubmit}
            projectIdmRoles={this.props.projectIdmRoles}
            projectFeatureFlags={this.props.projectFeatureFlags}
          />
          )}
      </div>
    );
  }
}

ProjectEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  project: PropTypes.object.isRequired,
  projectIdmRoles: PropTypes.array.isRequired,
  projectFeatureFlags: PropTypes.object.isRequired,
  loadProject: PropTypes.func.isRequired,
  saveProject: PropTypes.func.isRequired,
  clearProject: PropTypes.func.isRequired,
  getProjectIdmRoles: PropTypes.func.isRequired,
  getProjectFeatureFlags: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  project: state.projectData.project,
  projectIdmRoles: state.projectIdmRoles,
  projectFeatureFlags: state.projectFeatureFlags,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadProject,
  saveProject,
  clearProject,
  getProjectIdmRoles,
  getProjectFeatureFlags,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ProjectEditPage);

export const EditProjectPage = wrapSubProjectEditPage(connectedPage);
export const NewProjectPage = connectedPage;
