import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormButtons from '../../components/forms/FormButtons';
import FormInput from '../../components/forms/FormInput';
import { inputTargetValue } from '../../utils/inputs';
import ApiCallExample from '../../components/ApiCallExample';
import { TabPane, Tabs } from '../../components/Tabs';
import * as AclTables from './AclTables';
import FormElement from '../../components/forms/FormElement';
import FormCheckbox from '../../components/forms/FormCheckbox';
import AbcSuggest from '../../components/AbcSuggest/AbcSuggest';
import UserSuggest from '../../components/UserSuggest';
import FormMetricNameLabel from '../../components/forms/FormMetricNameLabel';
import IdmRolesTable from './IdmRolesTable';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';
import { isCloud } from '../../utils/env';

function deleteEntryReducer(state, index) {
  const newAcl = state.acl.slice();
  newAcl.splice(index, 1);
  return { acl: newAcl };
}

function updateEntryReducer(state, index, entry) {
  const newEntry = { ...entry, login: entry.login.trim() };

  const newAcl = state.acl.slice();
  if (index >= 0) {
    newAcl.splice(index, 1, newEntry);
  } else {
    newAcl.push(entry);
  }
  return { acl: newAcl };
}

class ProjectForm extends PureComponent {
  static projectToState = (project, isNew) => {
    if (project.id) {
      const state = { ...project };
      if (isNew) {
        state.id = '';
        state.name = '';
      }
      return state;
    }
    return { id: '', acl: [] };
  };

  constructor(props) {
    super(props);
    this.state = ProjectForm.projectToState(props.project, props.isNew);
  }

  componentDidUpdate(prevProps) {
    if (this.props.project !== prevProps.project && this.props.isNew !== prevProps.isNew) {
      this.state = ProjectForm.projectToState(this.props.project, false);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);
    this.setState({ [name]: value });
  };

  onOwnerChange = (value) => {
    this.setState({ owner: value });
  };

  onAbcServiceChange = (value) => {
    this.setState({ abcService: value });
  };

  onDeleteRow = (index) => {
    this.setState(deleteEntryReducer(this.state, index));
  };

  onMetricNameLabelChange = (value) => {
    this.setState({ metricNameLabel: value });
  };

  onUpdateRow = (index, row) => {
    this.setState(updateEntryReducer(this.state, index, row));
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.state);
  };

  onJsonStateChange = (newState) => {
    this.setState(newState);
  };

  render() {
    const { isNew } = this.props;

    const {
      id, name, owner, abcService, acl, onlyAuthRead,
    } = this.state;

    const monitoringUiHost = getMonitoringUiHost();

    return (
      <form className="form-horizontal">
        <div className="col-lg-6 col-md-7">
          <Tabs>
            <TabPane label="Generic">
              <FormInput
                type="text" name="id" label="Id" value={id} onChange={this.onInputChange}
                disabled={!isNew}
                help="Unique project identificator"
              />
              <FormInput
                type="text" name="name" label="Name" value={name} onChange={this.onInputChange}
                help="This value will be displayed in Solomon admin page"
              />
              {!isNew && (
                <FormElement
                  label="Owner"
                  help="Project owner"
                >
                  <UserSuggest value={owner} onChange={this.onOwnerChange} />
                </FormElement>
              )}
              <FormElement
                label="ABC service"
                help="ABC service slug for project"
              >
                <AbcSuggest value={abcService} onChange={this.onAbcServiceChange} />
              </FormElement>
              <FormCheckbox
                name="onlyAuthRead"
                label="Allow only authorized read"
                value={onlyAuthRead}
                onChange={this.onInputChange}
              />
              <FormMetricNameLabel
                type="project"
                name="metricNameLabel"
                label="Metric name label"
                onChange={this.onMetricNameLabelChange}
                onlyMetricNameShards={this.state.onlySensorNameShards}
              />
              <FormCheckbox
                name="onlySensorNameShards"
                label="Allow only metric name shards"
                value={this.state.onlySensorNameShards}
                onChange={this.onInputChange}
              />
              <FormButtons onSubmit={this.onSubmit} />
            </TabPane>
            {!this.props.isNew && (
              <TabPane label="User permissions">
                {!isCloud() ? (
                  <>
                    <div className="alert alert-info">
                      Roles editing has moved to
                      <a
                        style={{ marginLeft: 4 }}
                        href={`https://${monitoringUiHost}/projects/${this.props.project.id}/access`}
                        target="_blank"
                        rel="noreferrer"
                      >
                        Monitoring UI
                      </a>
                    </div>
                    <IdmRolesTable roles={this.props.projectIdmRoles} />
                  </>
                ) : (
                  <AclTables.EditableTable
                    acl={acl}
                    onDelete={this.onDeleteRow}
                    onUpdate={this.onUpdateRow}
                  />
                )}
                {!isCloud() ? null : (
                  <FormButtons onSubmit={this.onSubmit} />
                )}
              </TabPane>
            )}
          </Tabs>
        </div>
        <div className="col-lg-6 col-md-5">
          <ApiCallExample
            path="/projects"
            value={this.state}
            isNew={isNew}
            onChange={this.onJsonStateChange}
          />
        </div>
      </form>
    );
  }
}

ProjectForm.propTypes = {
  project: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  onSubmit: PropTypes.func.isRequired,
  projectIdmRoles: PropTypes.array.isRequired,
};

export default ProjectForm;
