import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import FormButtons from '../../components/forms/FormButtons';
import ApiCallExample from '../../components/ApiCallExample';
import MenuEditor from '../../components/MenuEditor/MenuEditor';

function cleanMenuItems(items) {
  // eslint-disable-next-line no-use-before-define
  return items ? items.map((x) => cleanMenuItem(x)) : [];
}

function cleanMenuItem(menuItem) {
  const newMenuItem = { ...menuItem };

  delete newMenuItem.expanded;
  if (!newMenuItem.url) {
    delete newMenuItem.url;
  }
  if (!newMenuItem.selectors) {
    delete newMenuItem.selectors;
  }
  if (!newMenuItem.children || newMenuItem.children.length === 0) {
    delete newMenuItem.children;
  } else {
    newMenuItem.children = cleanMenuItems(menuItem.children);
  }
  return newMenuItem;
}

class ProjectMenuForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { ...props.projectMenu, apiVisibility: true };
  }

  componentDidUpdate(prevProps) {
    if (this.props.projectMenu !== prevProps.projectMenu) {
      this.setState({ ...this.props.projectMenu, apiVisibility: true });
    }
  }

  onJsonStateChange = (newState) => {
    this.setState(newState);
  };

  onChangeMenuItems = (items) => {
    this.setState({ items });
  };

  onChangeApiVisibility = (apiVisibility) => {
    this.setState({ apiVisibility });
  };

  onSubmit = (event) => {
    event.preventDefault();
    const newProjectMenu = { ...this.state };
    newProjectMenu.items = cleanMenuItems(this.state.items);
    delete newProjectMenu.apiVisibility;
    this.props.onSubmit(newProjectMenu);
  };

  render() {
    const { projectId } = this.props;
    const { items } = this.state;

    const projectMenuJson = {
      items: cleanMenuItems(items),
      version: this.props.projectMenu.version,
    };

    return (
      <form className="form-horizontal">
        <div className={this.state.apiVisibility ? 'col-lg-6 col-md-7' : 'col-lg-10 col-md-11'}>
          <MenuEditor treeData={items} onChange={this.onChangeMenuItems} />
          <FormButtons onSubmit={this.onSubmit} />
        </div>
        <div className={this.state.apiVisibility ? 'col-lg-6 col-md-5' : 'col-lg-2 col-md-1'}>
          <ApiCallExample
            path={`/projects/${projectId}/menu`}
            value={projectMenuJson}
            onChange={this.onJsonStateChange}
            visible={this.state.apiVisibility}
            onChangeVisibility={this.onChangeApiVisibility}
            isNew
          />
        </div>
      </form>
    );
  }
}

ProjectMenuForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  projectMenu: PropTypes.object.isRequired,
  onSubmit: PropTypes.func.isRequired,
};

export default ProjectMenuForm;
