import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import ProjectMenuForm from './ProjectMenuForm';
import { wrapSubProjectEditPage } from './wrapSubProjectPage';

import { findProjectMenu, saveProjectMenu } from '../../store/reducers/projects/projectMenu';

import './ProjectsPage.css';

class ProjectMenuPage extends PureComponent {
  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
  }

  componentDidMount() {
    this.props.findProjectMenu(this._projectId);
  }

  componentDidUpdate(prevProps) {
    const { projectId } = this.props.match.params;
    const prevProjectId = prevProps.match.params.projectId;
    if (projectId !== prevProjectId) {
      this._projectId = projectId;
      this.props.findProjectMenu(projectId);
    }
  }

  onSubmit = (projectMenu) => {
    const { history } = this.props;
    const newProjectMenu = { ...projectMenu, id: this._projectId };
    this.props.saveProjectMenu(this._projectId, newProjectMenu)
      .then((action) => {
        if (action) {
          history.push(`/admin/projects/${this._projectId}`);
        }
      });
  };

  render() {
    const { projectMenu } = this.props;
    const isLoading = (isEmpty(projectMenu));

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Project menu of ${this.props.projectName}`} />
        {isLoading ? <span>Loading...</span> : (
          <ProjectMenuForm
            projectId={this._projectId}
            projectMenu={projectMenu}
            onSubmit={this.onSubmit}
          />
        )}
      </div>
    );
  }
}

ProjectMenuPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
  projectMenu: PropTypes.object.isRequired,
  findProjectMenu: PropTypes.func.isRequired,
  saveProjectMenu: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectName: state.projectData.project.name,
  projectMenu: state.projectMenu,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  findProjectMenu,
  saveProjectMenu,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ProjectMenuPage);

export default wrapSubProjectEditPage(connectedPage);
