import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../components/PageTitle';
import Panel from '../../components/Panel';
import LabeledValue from '../../components/LabeledValue';
import ShowActions from '../../components/ShowActions';
import UserName from '../../components/UserName';
import * as AclTables from './AclTables';
import IdmRolesTable from './IdmRolesTable';
import EntityInfo from '../../components/EntityInfo';

import { wrapSubProjectReadPage } from './wrapSubProjectPage';
import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

import { findProjectClusters } from '../../store/reducers/clusters/clustersPage';
import { findProjectServices } from '../../store/reducers/services/servicesPage';
import { findProjectShards } from '../../store/reducers/shards/shardsPage';
import { findProjectGraphs } from '../../store/reducers/graphs/graphsPage';
import { findProjectDashboards } from '../../store/reducers/dashboards/dashboardsPage';
import { clearProjectAlerts } from '../../store/reducers/alerts/alertsPage';
import { deleteProject } from '../../store/reducers/projects/projectData';
import { getProjectIdmRoles } from '../../store/reducers/projects/projectIdmRoles';
import { getProjectFeatureFlags } from '../../store/reducers/projects/projectFeatureFlags';
import { AdminYasmAlert } from '../../components/YasmAlert';
import { isCloud, isPreOrProdIntEnv } from '../../utils/env';
import FavoriteProjectToggler from './FavoriteProjectToggler';
import MonitoringButton from '../old/components/ControlBar/MonitoringButton';
import { getMonitoringUiHost } from '../old/utils/graphToExpressionTransformations';

const StatusCard = ({
  title, icon, color, link, value, detailsText,
}) => (
  <div className="col-lg-3 col-md-6">
    <Link to={link}>
      <div className={`panel panel-${color}`}>
        <div className="panel-heading">
          <div className="row">
            <div className="col-xs-3" style={{ fontSize: '4em' }}>
              <i className={`glyphicon glyphicon-${icon}`} />
            </div>
            <div className="col-xs-9 text-right">
              <div className="huge">{value === null ? '' : value}</div>
              <div>{title}</div>
            </div>
          </div>
        </div>
        <div className="panel-footer">
          <span className="pull-left">{detailsText}</span>
          <span className="pull-right">
            <i className="glyphicon glyphicon-option-horizontal" />
          </span>
          <div className="clearfix" />
        </div>
      </div>
    </Link>
  </div>
);

StatusCard.propTypes = {
  title: PropTypes.string.isRequired,
  icon: PropTypes.string.isRequired,
  color: PropTypes.oneOf(['primary', 'info', 'green', 'yellow', 'red']).isRequired,
  link: PropTypes.string.isRequired,
  value: PropTypes.oneOfType([PropTypes.number, PropTypes.string]),
  detailsText: PropTypes.string,
};

StatusCard.defaultProps = {
  detailsText: 'Configure',
  value: null,
};

const ProjectStatus = ({
  projectId, clustersCount, servicesCount, shardsCount, alertsCount,
}) => (
  <div className="row">
    <StatusCard
      title="Clusters"
      icon="equalizer"
      color="primary"
      link={`/admin/projects/${projectId}/clusters`}
      value={clustersCount}
    />
    <StatusCard
      title="Services"
      icon="cloud"
      color="green"
      link={`/admin/projects/${projectId}/services`}
      value={servicesCount}
    />
    <StatusCard
      title="Shards"
      icon="tasks"
      color="yellow"
      link={`/admin/projects/${projectId}/shards`}
      value={shardsCount}
    />
    <StatusCard
      title={alertsCount === undefined ? '' : 'Alerts'}
      icon="bullhorn"
      color="red"
      link={`/admin/projects/${projectId}/alerts`}
      value={alertsCount === undefined ? 'Alerts' : alertsCount}
    />
  </div>
);

ProjectStatus.propTypes = {
  projectId: PropTypes.string.isRequired,
  clustersCount: PropTypes.number.isRequired,
  servicesCount: PropTypes.number.isRequired,
  shardsCount: PropTypes.number.isRequired,
  alertsCount: PropTypes.number,
};

ProjectStatus.defaultProps = {
  alertsCount: undefined,
};

class ProjectPage extends PureComponent {
  componentDidMount() {
    const { match } = this.props;
    const { projectId } = match.params;
    this.props.findProjectServices(projectId);
    this.props.findProjectClusters(projectId);
    this.props.findProjectShards(projectId);
    this.props.findProjectGraphs(projectId);
    this.props.findProjectDashboards(projectId);
    this.props.clearProjectAlerts();
    this.props.getProjectIdmRoles(projectId);
    this.props.getProjectFeatureFlags(projectId);
  }

  onDelete = (event) => {
    event.preventDefault();
    const { match, history } = this.props;
    this.props.deleteProject(match.params.projectId)
      .then((action) => {
        if (action) {
          history.push('/admin/projects');
        }
      });
  };

  render() {
    const { projectId } = this.props.match.params;
    const {
      clustersPage, servicesPage, shardsPage, alertStats,
      project, projectAuth, projectIdmRoles,
    } = this.props;
    const clustersCount = clustersPage.page ? clustersPage.page.totalCount || 0 : 0;
    const servicesCount = servicesPage.page ? servicesPage.page.totalCount || 0 : 0;
    const shardsCount = shardsPage.page ? shardsPage.page.totalCount || 0 : 0;
    const alertsCount = alertStats.alertsCount === undefined ? undefined : alertStats.alertsCount;
    const acl = project.acl || [];

    const canChangeProject = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDeleteProject = projectAuth.isOwner() || Auth.hasRole(ADMIN);

    const subTitle = (
      <>
        <FavoriteProjectToggler projectId={projectId} />
        {isPreOrProdIntEnv() && (
          <span style={{ margin: '0 3ch' }}>
            <MonitoringButton
              href={`https://${getMonitoringUiHost()}/projects/${projectId}?utm_source=solomon_view_project`}
            />
          </span>
        )}
        <a href={`/?project=${projectId}`}>go to main UI</a>
      </>
    );

    let metricNameLabel = '';
    let metricNameLabelHint = '';
    if (project.metricNameLabel) {
      metricNameLabel = project.metricNameLabel;
    } else if (project.onlySensorNameShards) {
      metricNameLabel = isCloud() ? 'name' : 'sensor';
      metricNameLabelHint = 'default value for project with required metric name shards';
    }

    let authorizationTypeInfo = '';
    const authType = projectAuth.getAuthorizationType();
    switch (authType) {
      case 'IAM':
        authorizationTypeInfo = 'Using IAM authorization instead of Project ACL';
        break;
      case 'INTERNAL':
        authorizationTypeInfo = 'Using internal authorization instead of Project ACL';
        break;
      case 'PROJECT_ACL':
        authorizationTypeInfo = '';
        break;
      case 'ROLE':
        authorizationTypeInfo = 'Using role authorization instead of Project ACL';
        break;
      default:
        authorizationTypeInfo = `Using ${authType || 'unknown'} authorization instead of Project ACL`;
    }

    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Project ${project.name}`} subTitle={subTitle} />
        <AdminYasmAlert projectId={projectId} />
        <ProjectStatus
          projectId={projectId}
          clustersCount={clustersCount}
          servicesCount={servicesCount}
          shardsCount={shardsCount}
          alertsCount={alertsCount}
        />
        <div className="row">
          <div className="col-xs-6">
            <Panel title="Generic options">
              <LabeledValue label="Id" value={projectId} />
              <LabeledValue label="Owner" value={<UserName login={project.owner || ''} />} />
              {project.abcService && (
                <LabeledValue
                  label="ABC service"
                  value={(
                    <a
                      href={`https://abc.yandex-team.ru/services/${project.abcService}`}
                      target="_blank"
                      rel="noopener noreferrer"
                    >
                      {project.abcService}
                    </a>
                  )}
                />
              )}
              <EntityInfo entity={project} />
              {metricNameLabel ? (
                <LabeledValue
                  label="Metric name label"
                  value={metricNameLabel}
                  hint={metricNameLabelHint}
                />
              ) : null}
              {project.onlySensorNameShards && (
                <LabeledValue
                  label=""
                  value={(
                    <span>
                      <i className="glyphicon glyphicon-warning-sign" />
                      {' '}
                      Metric name is required for services or shards
                    </span>
                  )}
                />
              )}
              {project.onlyNewFormatWrites && (
                <LabeledValue
                  label=""
                  value={(
                    <span>
                      <i className="glyphicon glyphicon-warning-sign" />
                      {' '}
                      Allow only writes in new format
                    </span>
                  )}
                />
              )}
              {project.onlyNewFormatReads && (
                <LabeledValue
                  label=""
                  value={(
                    <span>
                      <i className="glyphicon glyphicon-warning-sign" />
                      {' '}
                      Allow only reads in new format
                    </span>
                  )}
                />
              )}
              {project.onlyAuthRead && (
                <LabeledValue
                  label=""
                  value={(
                    <span>
                      <i className="glyphicon glyphicon-warning-sign" />
                      {' '}
                      Allow only authorized read
                    </span>
                  )}
                />
              )}
              <LabeledValue label="" value={<Link to={`/admin/projects/${projectId}/status`}>Project status</Link>} />
              {authorizationTypeInfo ? <LabeledValue label="" value={authorizationTypeInfo} /> : null}
            </Panel>
            <ShowActions
              editPath={`/admin/projects/${projectId}/edit`}
              duplicatePath={`/admin/projects/new?duplicateOf=${project.id}`}
              canEdit={canChangeProject}
              canDelete={canDeleteProject}
              canChangeState={false}
              canDuplicate={!isCloud()}
              onDeactivate={this.onDeactivate}
              onActivate={this.onActivate}
              onDelete={this.onDelete}
            />
          </div>
          <div className="col-xs-6">
            {!isCloud() ? (
              <Panel title="Roles">
                <IdmRolesTable roles={projectIdmRoles} limit={5} />
              </Panel>
            ) : (
              <Panel title="User permissions">
                <AclTables.ReadonlyTable acl={acl} limit={5} />
              </Panel>
            )}
          </div>
        </div>
      </div>
    );
  }
}

ProjectPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  clustersPage: PropTypes.object.isRequired,
  servicesPage: PropTypes.object.isRequired,
  shardsPage: PropTypes.object.isRequired,
  alertStats: PropTypes.object.isRequired,
  project: PropTypes.object.isRequired,
  projectAuth: PropTypes.object.isRequired,
  projectIdmRoles: PropTypes.array.isRequired,
  deleteProject: PropTypes.func.isRequired,
  findProjectClusters: PropTypes.func.isRequired,
  findProjectServices: PropTypes.func.isRequired,
  findProjectShards: PropTypes.func.isRequired,
  findProjectGraphs: PropTypes.func.isRequired,
  findProjectDashboards: PropTypes.func.isRequired,
  clearProjectAlerts: PropTypes.func.isRequired,
  getProjectIdmRoles: PropTypes.func.isRequired,
  getProjectFeatureFlags: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  clustersPage: state.clustersPage,
  servicesPage: state.servicesPage,
  shardsPage: state.shardsPage,
  alertStats: state.projectAlertStats,
  projectIdmRoles: state.projectIdmRoles,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  deleteProject,
  findProjectClusters,
  findProjectServices,
  findProjectShards,
  findProjectGraphs,
  findProjectDashboards,
  clearProjectAlerts,
  getProjectIdmRoles,
  getProjectFeatureFlags,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(ProjectPage);

export default wrapSubProjectReadPage(connectedPage);
