/* eslint-disable no-restricted-syntax */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';

import PageTitle from '../../components/PageTitle';
import { wrapSubProjectReadPage } from './wrapSubProjectPage';
import { TabPane } from '../../components/Tabs';
import TabsWithUrl from '../../components/TabsWithUrl';
import Breadcrumb from '../../components/Breadcrumb/Breadcrumb';
import {
  projectForEnv,
  coreClusterForEnv,
  alertingClusterForEnv,
  knownDcs,
  hasProcessingShardDataForEnv,
} from '../../utils/env';
import { formatSearch, getUrlHash, setUrlHash } from '../../utils/url';
import SolomonDash from '../../components/SolomonDash/SolomonDash';
import { convertColumnsToRows } from '../../components/SolomonDash/convertColumnsToRows';
import DashboardTimeSelector from '../../components/SolomonDash/DashboardTimeSelector';

class ProjectStatusPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      dashboardDuration: '1h',
    };
  }

  onDashboardDurationChange = (dashboardDuration) => {
    this.setState({ dashboardDuration });
  };

  _dashboardSettingsRow() {
    return (
      <DashboardTimeSelector
        duration={this.state.dashboardDuration}
        onChange={this.onDashboardDurationChange}
      />
    );
  }

  _targetsStatusGraphs() {
    const { projectId } = this.props.match.params;

    const columns = [];

    for (const dc of knownDcs) {
      const dcTitle = dc.title;
      const dcHost = dc.host;
      const fetchingStatusUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'fetcher',
        'l.host': dcHost,
        'l.sensor': 'fetch.urlStatus',
        'l.projectId': projectId,
        'l.shardId': 'total',
        'l.type': '*',
        graph: 'auto',
      })}`;

      const responseSizeUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'fetcher',
        'l.host': dcHost,
        'l.sensor': 'fetch.responseByteSize',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const urlsCountUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'fetcher',
        'l.host': dcHost,
        'l.sensor': 'fetch.urlCount',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const downloadTimeUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'fetcher',
        'l.host': dcHost,
        'l.sensor': 'fetch.downloadMillis',
        'l.projectId': projectId,
        'l.shardId': 'total',
        'l.bin': '*',
        graph: 'auto',
      })}`;

      const columnItems = [
        { title: 'Fetching Status', url: fetchingStatusUrl },
        { title: 'Response Size', url: responseSizeUrl },
        { title: 'Urls Count', url: urlsCountUrl },
        { title: 'Download time', url: downloadTimeUrl },
      ];

      columns.push({ name: dcTitle, items: columnItems });
    }

    const rows = convertColumnsToRows(columns);

    return (
      <div>
        {this._dashboardSettingsRow()}
        <SolomonDash
          id="targets_status_graphs"
          rows={rows}
          duration={this.state.dashboardDuration}
        />
      </div>
    );
  }

  _coremonStatusGraphs() {
    const { projectId } = this.props.match.params;

    const columns = [];

    for (const dc of knownDcs) {
      const dcTitle = dc.title;
      const dcHost = dc.host;
      const sensorsParseErrorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.sensorParseError',
        'l.type': '*',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const urlProcessResultUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.urlProcessResult',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const processSensorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.sensorsTotal',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const writersToStorageUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.stockpileWrites.completed*',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const columnItems = [
        { title: 'Metrics parse errors', url: sensorsParseErrorsUrl },
        { title: 'Url process result', url: urlProcessResultUrl },
        { title: 'Process metrics/s', url: processSensorsUrl },
        { title: 'Writers To Storage', url: writersToStorageUrl },
      ];

      columns.push({ name: dcTitle, items: columnItems });
    }

    const rows = convertColumnsToRows(columns);

    return (
      <div>
        {this._dashboardSettingsRow()}
        <div className="row">
          <SolomonDash
            id="coremon_status_graphs"
            rows={rows}
            duration={this.state.dashboardDuration}
          />
        </div>
      </div>
    );
  }

  _sensorsCountGraphs() {
    const { projectId } = this.props.match.params;

    const columns = [];

    for (const dc of knownDcs) {
      const dcTitle = dc.title;
      const dcHost = dc.host;
      const fileSensorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.fileSensors',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const inMemorySensorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.inMemSensors',
        'l.projectId': projectId,
        'l.shardId': 'total',
        graph: 'auto',
      })}`;

      const topShardsByFileSensorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.fileSensors',
        'l.projectId': projectId,
        'l.shardId': '!total',
        graph: 'auto',
        filter: 'top',
        filterLimit: '30',
        graphMode: 'pieChart',

      })}`;

      const topShardsByInMemorySensorsUrl = `/?${formatSearch({
        project: projectForEnv,
        cluster: coreClusterForEnv,
        service: 'coremon',
        'l.host': dcHost,
        'l.sensor': 'engine.inMemSensors',
        'l.projectId': projectId,
        'l.shardId': '!total',
        graph: 'auto',
        filter: 'top',
        filterLimit: '30',
        graphMode: 'pieChart',
      })}`;

      const columnItems = [
        { title: 'File metrics', url: fileSensorsUrl },
        { title: 'In-memory metrics', url: inMemorySensorsUrl },
        { title: 'TOP-30 shards by file metrics count', url: topShardsByFileSensorsUrl },
        { title: 'TOP-30 shards by in-memory metrics count', url: topShardsByInMemorySensorsUrl },
      ];

      columns.push({ name: dcTitle, items: columnItems });
    }

    const rows = convertColumnsToRows(columns);

    return (
      <div>
        {this._dashboardSettingsRow()}
        <div className="row">
          <SolomonDash
            id="sensors_count_graphs"
            rows={rows}
            duration={this.state.dashboardDuration}
          />
        </div>
      </div>
    );
  }

  _pushStatusGraphs() {
    const { projectId } = this.props.match.params;

    const pushAuthTypesUrl = `/?${formatSearch({
      project: projectForEnv,
      cluster: coreClusterForEnv,
      service: 'gateway',
      'l.host': 'cluster',
      'l.sensor': 'push.shard.request.count',
      'l.projectId': projectId,
      'l.shardId': 'total',
      'l.authType': '!total',
      graph: 'auto',
    })}`;

    const pushStatusUrl = `/?${formatSearch({
      project: projectForEnv,
      cluster: coreClusterForEnv,
      service: 'gateway',
      'l.host': 'cluster',
      'l.sensor': 'push.shard.requests.total',
      'l.projectId': projectId,
      'l.shardId': 'total',
      'l.sensorsFormat': 'total',
      'l.pushStatus': '!NOT_OK',
      graph: 'auto',
    })}`;

    const pushProcessedSensorsUrl = `/?${formatSearch({
      project: projectForEnv,
      cluster: coreClusterForEnv,
      service: 'gateway',
      'l.host': 'cluster',
      'l.sensor': 'push.shard.sensorsProcessed',
      'l.projectId': projectId,
      'l.shardId': 'total',
      'l.sensorsFormat': 'total',
      graph: 'auto',
    })}`;

    const rows = [
      {
        items: [
          { title: 'Authentication request types', url: pushAuthTypesUrl },
        ],
      },
      {
        items: [
          { title: 'Push status', url: pushStatusUrl },
        ],
      },
      {
        items: [
          { title: 'Push processed metrics, /s', url: pushProcessedSensorsUrl },
        ],
      },
    ];

    return (
      <div>
        {this._dashboardSettingsRow(false)}
        <SolomonDash
          id="push_status_graphs"
          rows={rows}
          duration={this.state.dashboardDuration}
        />
      </div>
    );
  }

  _alertingStatusGraphs() {
    const { projectId } = this.props.match.params;

    const rows = [
      {
        items: [
          {
            title: 'OK',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=OK&graph=auto&graphOnly=1`,
          },
          {
            title: 'ALARM',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=ALARM&graph=auto&graphOnly=1`,
          },
          {
            title: 'NO_DATA',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=NO_DATA&graph=auto&graphOnly=1`,
          },
          {
            title: 'ERROR',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=ERROR&graph=auto&graphOnly=1`,
          },
          {
            title: 'WARN',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=WARN&graph=auto&graphOnly=1`,
          },
          {
            title: 'Evaluation status',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.status.total&l.projectId=${projectId}&l.status=*&graph=auto&b=1h&graphMode=pieChart&graphOnly=1`,
          },
        ],
      },
      {
        items: [
          {
            title: 'Evaluation/sec',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=!total&l.host=cluster&l.sensor=evaluations.eval.started&l.projectId=${projectId}&graph=auto&transform=differentiate&graphOnly=1`,
            colspan: 3,
          },
          {
            title: 'Count evaluations in-flight',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=!total&l.host=cluster&l.sensor=evaluations.eval.inFlight&l.projectId=${projectId}&graph=auto&graphOnly=1`,
            colspan: 3,
          },
        ],
      },
      {
        items: [
          {
            title: 'Evaluation elapsed time',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.bin=*&l.host=cluster&l.sensor=evaluations.eval.elapsedTimeMillis&l.projectId=${projectId}&graph=auto&cs=gradient&graphOnly=1`,
            colspan: 3,
          },
          {
            title: 'Evaluation lag lagSeconds',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.alertType=total&l.host=cluster&l.sensor=evaluations.eval.lagSeconds&l.projectId=${projectId}&l.bin=*&graph=auto&cs=gradient&graphOnly=1`,
            colspan: 3,
          },
        ],
      },
      {
        items: [
          {
            title: 'Notifications/sec',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.host=cluster&l.sensor=notifications.channel.notify.started&l.projectId=${projectId}&l.type=!total&graph=auto&transform=differentiate&graphOnly=1`,
            colspan: 3,
          },
          {
            title: 'Count notifications in-flight',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.host=cluster&l.sensor=notifications.channel.notify.inFlight&l.projectId=${projectId}&l.type=!total&graph=auto&graphOnly=1`,
            colspan: 3,
          },
        ],
      },
      {
        items: [
          {
            title: 'Notification elapsed time',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.host=cluster&l.sensor=notifications.channel.notify.elapsedTimeMillis&l.type=total&l.projectId=${projectId}&graph=auto&cs=gradient&graphOnly=1`,
            colspan: 3,
          },
          {
            title: 'Notification lag seconds',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.bin=*&l.host=cluster&l.sensor=notifications.channel.notify.lagSeconds&l.type=total&l.projectId=${projectId}&graph=auto&cs=gradient&graphOnly=1`,
            colspan: 3,
          },
        ],
      },
      {
        items: [
          {
            title: 'Success notifications',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.host=cluster&l.sensor=notifications.channel.notify.status.total&l.projectId=${projectId}&l.status=SUCCESS&graph=auto&transform=differentiate&l.type=!total&graphOnly=1`,
            colspan: 3,
          },
          {
            title: 'Failed notifications',
            url: `/?project=${projectForEnv}&cluster=${alertingClusterForEnv}&service=alerting&l.host=cluster&l.sensor=notifications.channel.notify.status.total&l.type=total&l.projectId=${projectId}&l.status=!SUCCESS|SKIP*&graph=auto&transform=differentiate&graphOnly=1`,
            colspan: 3,
          },
        ],
      },
    ];

    return (
      <div>
        {this._dashboardSettingsRow(false)}
        <SolomonDash
          id="alerting_status_graphs"
          rows={rows}
          duration={this.state.dashboardDuration}
        />
      </div>
    );
  }

  render() {
    return (
      <div>
        <Breadcrumb match={this.props.match} />
        <PageTitle title={`Status of ${this.props.projectName}`} />
        <TabsWithUrl activeTabId={getUrlHash()} setActiveTab={setUrlHash}>
          <TabPane label="Fetching">
            {this._targetsStatusGraphs()}
          </TabPane>
          <TabPane label="Push">
            {this._pushStatusGraphs()}
          </TabPane>
          {hasProcessingShardDataForEnv && (
            <TabPane label="Processing">
              {this._coremonStatusGraphs()}
            </TabPane>
          )}
          <TabPane label="Metrics count">
            {this._sensorsCountGraphs()}
          </TabPane>
          <TabPane label="Alerting">
            {this._alertingStatusGraphs()}
          </TabPane>
        </TabsWithUrl>
      </div>
    );
  }
}

ProjectStatusPage.propTypes = {
  match: PropTypes.object.isRequired,
  projectName: PropTypes.string.isRequired,
};

export default wrapSubProjectReadPage(connect(
  (state) => ({ projectName: state.projectData.project.name }),
)(ProjectStatusPage));
