import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import { fetchProjects } from '../../store/reducers/projects/projects';

import PageTitle from '../../components/PageTitle';
import RadioButtons from '../../components/RadioButtons';

import {
  CONFIG_DELETE,
  CONFIG_UPDATE,
} from '../../auth/ProjectPermissions';

import './ProjectsPage.css';
import ProjectsBlock from './ProjectsBlock';
import { parseSearch } from '../../utils/url';

const PM_ALL = 'ALL';
const PM_ALL_TITLE = 'All';

const PM_ADMIN = 'ADMIN';
const PM_ADMIN_TITLE = 'Admin';

const ADMIN_PERMISSIONS = [CONFIG_UPDATE, CONFIG_DELETE].join(',');

const mapStateToParams = (state) => {
  const params = {};
  const {
    text, permissionMode, page, pageSize,
  } = state;
  if (text) {
    params.text = text;
  }
  if (page !== 0) {
    params.page = page;
  }

  params.pageSize = pageSize;

  if (permissionMode === PM_ADMIN) {
    params.filterByPermissions = ADMIN_PERMISSIONS;
  }

  return params;
};

class ProjectsSelectForMultipleAlertsPage extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      text: '', page: 0, pageSize: 50, permissionMode: PM_ALL,
    };
  }

  componentDidMount() {
    this.projectFilter.focus();
    this.props.fetchProjects(mapStateToParams(this.state));
  }

  onFilterChange = (event) => {
    const text = event.target.value;
    this.doReload({ text });
  };

  onPermissionModeChange = (value) => {
    this.doReload({ permissionMode: value });
  };

  onActivePageChange = (activePage) => {
    this.doReload({ page: activePage - 1 });
  };

  onPageSizeChange = (pageSize) => {
    this.doReload({ pageSize, page: 0 });
  };

  makeProjectPath = (projectId) => {
    const parsedSearch = parseSearch(this.props.location.search);
    const serviceProviderId = parsedSearch.get('serviceProviderId') || '';
    const resourcesParam = parsedSearch.getAll('r');
    const resourcesStr = resourcesParam
      ? resourcesParam.join('&r=')
      : '';
    return `/admin/projects/${projectId}/alerts/templates/multipleResources?serviceProviderId=${serviceProviderId}&r=${resourcesStr}`;
  }

  doReload(change) {
    this.setState(change, () => {
      this.props.fetchProjects(mapStateToParams(this.state));
    });
  }

  render() {
    const { projectsPage, favoriteProjects } = this.props;

    const permissionModeSwitcher = (
      <div className="btn-group pull-right">
        <RadioButtons
          choices={[PM_ALL, PM_ADMIN]}
          labels={[PM_ALL_TITLE, PM_ADMIN_TITLE]}
          selected={this.state.permissionMode}
          onSelect={this.onPermissionModeChange}
        />
      </div>
    );

    return (
      <div>
        <PageTitle title="Select project for multiple alert creation" />

        <div className="row">
          <div className="btn-toolbar table-toolbar">
            <div className="btn-group">
              <input
                type="text"
                className="form-control"
                value={this.state.text}
                onChange={this.onFilterChange}
                ref={(input) => {
                  this.projectFilter = input;
                }}
                placeholder="Type to filter..."
              />
            </div>
            {permissionModeSwitcher}
          </div>
        </div>

        <div className="row">
          <ProjectsBlock
            projectsPage={projectsPage}
            makeProjectPath={this.makeProjectPath}
            favoriteProjects={favoriteProjects}
            text={this.state.text}
            isAdmin
            onActivePageChange={this.onActivePageChange}
            onPageSizeChange={this.onPageSizeChange}
          />
        </div>
      </div>
    );
  }
}

ProjectsSelectForMultipleAlertsPage.propTypes = {
  projectsPage: PropTypes.object.isRequired,
  favoriteProjects: PropTypes.array.isRequired,
  fetchProjects: PropTypes.func.isRequired,
  location: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  projectsPage: state.projects,
  favoriteProjects: state.favoriteProjects.projects,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  fetchProjects,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(ProjectsSelectForMultipleAlertsPage);
