import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import PermissionAlert from './alerts/PermissionAlert';
import WarningAlert from './alerts/WarningAlert';

import { loadProject } from '../../store/reducers/projects/projectData';
import { PROJECT_PERMISSIONS } from '../../auth/ProjectPermissions';

export const wrapSubProjectPage = (expectedPermission = null) => (Page) => {
  class SubProjectPage extends PureComponent {
    componentDidMount() {
      const { projectId } = this.props.match.params;

      const { project } = this.props.projectData;
      const prevProjectId = isEmpty(project) ? '' : project.id;

      if (projectId !== prevProjectId) {
        this.props.loadProject(projectId);
      }
    }

    componentDidUpdate(prevProps) {
      const curProjectId = this.props.match.params.projectId;
      const prevProjectId = prevProps.match.params.projectId;

      if (curProjectId !== prevProjectId) {
        this.props.loadProject(curProjectId);
      }
    }

    render() {
      const { projectId } = this.props.match.params;
      const {
        loading, project, projectAuth, error,
      } = this.props.projectData;

      if (loading) {
        return <div>Loading...</div>;
      }

      if (error) {
        if (error.code === 403) {
          return (
            <PermissionAlert message={error.message} owner={error.owner} />
          );
        }

        if (error.message === `no project with id '${projectId}'` && this.props.location.pathname === `/admin/projects/${projectId}/alerts/templates/multipleResources`) {
          window.location.replace(`/admin/selectProjectForMultipleAlerts${this.props.location.search}`);
          return null;
        }

        if (error.message) {
          return (
            <WarningAlert title="Failed to load project!" message={error.message} />
          );
        }

        return <WarningAlert title="Warning!" message="Failed to load project" />;
      }

      if (!projectAuth || !project) {
        return null;
      }

      if (!projectAuth.isAuthorizedFor(expectedPermission)) {
        const permissionMessage = expectedPermission
          ? expectedPermission.toLowerCase()
          : PROJECT_PERMISSIONS.READ.toLowerCase();
        const message = `You have no permissions to ${permissionMessage} in project ${projectId}`;
        return (<PermissionAlert message={message} owner={project.owner} />);
      }

      const newProps = { ...this.props, project, projectAuth };

      return <Page {...newProps} />;
    }
  }

  SubProjectPage.propTypes = {
    match: PropTypes.object.isRequired,
    projectData: PropTypes.object.isRequired,
    loadProject: PropTypes.func.isRequired,
    location: PropTypes.object.isRequired,
  };

  return connect(
    (state) => ({ projectData: state.projectData }),
    (dispatch) => bindActionCreators({
      loadProject,
    }, dispatch),
  )(SubProjectPage);
};

export const wrapSubProjectReadPage = wrapSubProjectPage();

export const wrapSubProjectEditPage = wrapSubProjectPage(PROJECT_PERMISSIONS.CONFIG_UPDATE);
